// backend/src/socket/socket.ts
import { Server, Socket } from "socket.io";
import jwt, { JwtPayload } from "jsonwebtoken";
import dotenv from "dotenv";

dotenv.config();

const SECRET_KEY = process.env.JWT_SECRET!;
let ioInstance: Server | null = null; // Initialize as null
const userSocketMap: { [userId: string]: string } = {}; // Map user IDs to socket IDs

export function initSocket(io: Server) {
  ioInstance = io;

  io.on("connection", (socket: Socket) => {
    console.log("🔌 Socket connected:", socket.id);

    const token = socket.handshake.auth?.token;

    if (!token) {
      console.log("❌ No token provided, disconnecting socket");
      socket.disconnect();
      return;
    }

    try {
      const decoded = jwt.verify(token, SECRET_KEY) as JwtPayload;

      if (!decoded || !decoded.userId || !decoded.role) {
        console.log("❌ Invalid token payload, disconnecting socket");
        socket.disconnect();
        return;
      }

      const userId = decoded.userId;
      const role = decoded.role;

      // Store the socket ID associated with the user ID
      userSocketMap[userId] = socket.id;
      console.log(`User ${userId} connected with socket ID: ${socket.id}`);
      // Join the role-based room (e.g., "admin", "agent", "user")
      socket.join(role);
      // Join user-specific room
      socket.join(`user_${userId}`);

      console.log(`✅ Socket ${socket.id} joined rooms: [${role}], [user_${userId}]`);

      socket.on("disconnect", () => {
        delete userSocketMap[userId]; // Remove the mapping on disconnect
        console.log(`🔌 Socket disconnected: ${socket.id}`);
      });
    } catch (err) {
      console.error("❌ Socket authentication error:", err);
      socket.disconnect();
    }
  });
}

// To emit notifications anywhere
export function getIO(): Server {
  if (!ioInstance) throw new Error("Socket.io not initialized");
  return ioInstance;
}