import { LeadStage, LeadStatus } from "@prisma/client";
import { Lead } from "../../interfaces/Lead";
import { LeadRepository } from "../../repositories/admin/leadRepository";

export class LeadService {
    private leadRepository: LeadRepository;

    constructor(leadRepository: LeadRepository) {
        this.leadRepository = leadRepository;
    }

    public async createLead(leadData: Lead, userName:string): Promise<Lead> {
        try {
            const newLead = await this.leadRepository.createLead(leadData,userName);
            return newLead;
        } catch (error) {
            console.error('Error creating lead in service:', error);
            throw error;
        }
    }

    public async getAllLeads(): Promise<Lead[]> {
        try {
            const leads = await this.leadRepository.getAllLeads();
            return leads;
        } catch (error) {
            console.error('Error getting leads in service:', error);
            throw error;
        }
    }

    public async getLeadById(leadId: string): Promise<Lead | null> {
        try {
            const lead = await this.leadRepository.getLeadById(leadId);
            return lead;
        } catch (error) {
            console.error('Error getting lead by ID in service:', error);
            throw error;
        }
    }

    public async updateLead(leadId: string, leadData: Partial<Lead>): Promise<Lead | null> {
        try {
            const updatedLead = await this.leadRepository.updateLead(leadId, leadData);
            return updatedLead;
        } catch (error) {
            console.error('Error updating lead in service:', error);
            throw error;
        }
    }

    public async deleteLead(leadId: string): Promise<void> {
        try {
            await this.leadRepository.deleteLead(leadId);
        } catch (error) {
            console.error('Error deleting lead in service:', error);
            throw error;
        }
    }

    public async fetchLeads(timeframe: string): Promise<any[]> {
        try {
            return await this.leadRepository.getLeadsByTimeframe(timeframe);
        } catch (error) {
            console.error("Error fetching leads in service:", error);
            throw error;
        }
    }
    public async changeStatus(id: string, status: LeadStatus): Promise<Lead | null> {
        return await this.leadRepository.updateLeadStatus(id, status);
      }
      
      public async changeStage(
        id: string,
        stage: LeadStage,
        userName?: string,
        notes?: string
      ): Promise<Lead | null> {
        return await this.leadRepository.updateLeadStage(id, stage, userName, notes);
      }

      public async assignLeadToOperations(leadId: string, operationUserId: string) {
        try {
          // You can add validations or checks here if needed
          return await this.leadRepository.assignLeadToOperations(leadId, operationUserId);
        } catch (error) {
          console.error('Service Error while assigning lead to operations:', error);
          throw new Error('Unable to assign lead to operations.');
        }
      }

            public async assignLeadToAssessment(leadId: string, assessmentUserId: string) {
        try {
            // You can add validations or checks here if needed
            return await this.leadRepository.assignLeadToAssessment(leadId, assessmentUserId);
            } catch (error) {
          console.error('Service Error while assigning lead to assessment:', error);    
            throw new Error('Unable to assign lead to assessment.');
        }
        }


      public async getQuotationAcceptedLeads() {
        try {
          return await this.leadRepository.findQuotationAcceptedLeads();
        } catch (error) {
          console.error("Error in getQuotationAcceptedLeads service:", error);
          throw new Error("Failed to retrieve quotation accepted leads");
        }
      }
      
      
}
