// backend/src/routes/adminRoutes.ts

import express, { NextFunction, Request, Response } from "express";
import { adminLogin } from "../controllers/admin/adminController";
import { refreshAccessToken } from "../controllers/authController";
import { authenticateUser } from "../middlewares/authMiddleware";
import { authorizeRole } from "../middlewares/roleMiddleware";
// Import all controllers, repositories, and services
import { getDashboardData, getLeadFunnel, getTopAgents, getTopSales } from "../controllers/admin/dashboardController";
import { agentUpload, packageUpload, documentUpload } from "../config/multerConfig";
import { AttractionController } from "../controllers/admin/attractionController";
import { HotelController } from "../controllers/admin/hotelController";
import { LeadController } from "../controllers/admin/leadController";
import { PackageController } from "../controllers/admin/packageController";
import { QuotationController } from "../controllers/admin/quotationController";
import { RestaurantController } from "../controllers/admin/restaurantController";
import { SupplierController } from "../controllers/admin/supplierController";
import { TransferController } from "../controllers/admin/transferController";
import { AgentController } from "../controllers/agent/agentController";
import { UserController } from "../controllers/admin/userController";
import { HotelRepository } from "../repositories/admin/hotelRepository";
import { LeadRepository } from "../repositories/admin/leadRepository";
import { PackageRepository } from "../repositories/admin/packageRepository";
import { QuotationRepository } from "../repositories/admin/quotationRepository";
import { TeamRepository } from "../repositories/admin/teamRepository";
import { AgentRepository } from "../repositories/agent/agentRepository";
import { HotelService } from "../services/admin/hotelService";
import { LeadService } from "../services/admin/leadService";
import { PackageService } from "../services/admin/packageService";
import { QuotationService } from "../services/admin/quotationService";
import { agentService } from "../services/agent/agentService";
import { getAdminNotifications, markAllNotificationsAsRead, markNotificationAsRead } from "../controllers/common/notificationController";
import { DocumentController } from "../controllers/admin/documentController";
import { DocumentService } from "../services/admin/documentService";
import { DocumentRepository } from "../repositories/admin/documentRepository";
import { GuideController } from "../controllers/admin/guideController"; // Import GuideController
import { TourRepository } from "../repositories/admin/tourRepository";
import { TourService } from "../services/admin/tourService";
import { TourController } from "../controllers/admin/tourController";
import {BookingRepository} from "../repositories/admin/bookingRepository";
import { BookingService } from "../services/admin/bookingService";
import { BookingController } from "../controllers/admin/bookingController";

// import { cloudConfigDocument, uploadToCloudDocument } from "../../config/cloudStorageDocument";

// Initialize Router
const adminRouter = express.Router();

// Initialize Controllers and Services
const hotelRepository = new HotelRepository();
const hotelService = new HotelService(hotelRepository);
const hotelController = new HotelController(hotelService);
const restaurantController = new RestaurantController();
const agentRepository = new AgentRepository();
const agentservice = new agentService(agentRepository);
const agentController = new AgentController(agentservice);
const supplierController = new SupplierController();
const transferController = new TransferController();
const attractionController = new AttractionController();
const userController = new UserController();
const teamRepository = new TeamRepository();
const leadRepository = new LeadRepository();
const leadService = new LeadService(leadRepository);
const leadController = new LeadController(leadService);
const packageRepository = new PackageRepository();
const packageService = new PackageService(packageRepository);
const packageController = new PackageController(packageService);
const bookingRepository = new BookingRepository();
const bookingService = new BookingService();
const bookingController = new BookingController(bookingService);
const quotationRepository = new QuotationRepository();
const quotationService = new QuotationService(quotationRepository);
const quotationController = new QuotationController(quotationService,bookingService);

const documentRepository = new DocumentRepository();
const documentService = new DocumentService(documentRepository);
const documentController = new DocumentController(documentService);
const guideController = new GuideController(); // Create an instance of GuideController

const tourRepository = new TourRepository();
const tourService = new TourService();
const tourController = new TourController();




// ==============================================
// AUTHENTICATION ROUTES
// ==============================================
adminRouter.post("/login", adminLogin);


adminRouter.get("/",  getDashboardData);
adminRouter.get("/top-agents", authenticateUser,
  authorizeRole(["admin"]), getTopAgents);
adminRouter.get("/top-sales", getTopSales);
adminRouter.get("/lead-funnel", 
  (req, res, next) => {
    getLeadFunnel(req, res).catch(next);
  });
adminRouter.post("/refresh-token", refreshAccessToken);
// ==============================================
// HOTEL MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createHotel",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => hotelController.createHotel(req, res)
);
adminRouter.get("/hotels", authenticateUser,
  authorizeRole(["admin", "user", "agent"]), (req, res, next) =>
  hotelController.getHotels(req, res, next)
);
adminRouter.get("/getHotel/:id",  authenticateUser,
  authorizeRole(["admin", "user", "agent"]), (req, res, next) =>
  hotelController.getHotelById(req, res, next)
);
adminRouter.put(
  "/updateHotel/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res, next) => hotelController.updateHotel(req, res, next)
);
adminRouter.delete(
  "/deleteHotel/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res, next) => hotelController.deleteHotel(req, res, next)
);
adminRouter.post(
  "/createRoom/:hotelId",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => hotelController.createRoom(req, res)
);

adminRouter.put(
  "/updateRoom/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => hotelController.updateRoom(req, res)
);
adminRouter.delete(
  "/deleteRoom/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => hotelController.deleteRoom(req, res)
);
adminRouter.get("/getRoom/:id",authenticateUser,
  authorizeRole(["admin", "user", "agent"]), (req, res) => 
  hotelController.getRoomById(req, res)
);



adminRouter.get("/getRooms/:hotelId", authenticateUser,
  authorizeRole(["admin","user","agent"]), (req, res) =>
  hotelController.getRoomsByHotel(req, res)
);
// ==============================================
// AGENT MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/agents/createAgent",
  authenticateUser,
  authorizeRole(["admin"]),
  agentUpload.single("image"), // Apply multer middleware here
  (req: Request, res: Response, next: NextFunction) =>
    agentController.createAgent(req, res, next)
);
adminRouter.put(
  "/agents/updateAgent/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  agentUpload.single("image"), // Apply multer middleware here
  (req: Request, res: Response, next: NextFunction) =>
    agentController.updateAgent(req, res, next)
);
adminRouter.get("/agents",authenticateUser,
authorizeRole(["admin", "user"]), (req: Request, res: Response, next: NextFunction) =>
  agentController.getAllAgents(req, res, next)
);
adminRouter.delete(
  "/agents/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentController.softDeleteAgent(req, res, next);
    console.log("delete", req.params.id);
  }
);
adminRouter.put(
  "/agents/restore/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentController.restoreAgent(req, res, next);
  }
);
adminRouter.delete(
  "/agents/permenantDelete/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) => {
    agentController.deleteAgent(req, res, next);
  }
);
// ==============================================
// SUPPLIER MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/suppliers",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => supplierController.createSupplier(req, res)
);
adminRouter.get("/suppliers",authenticateUser,
 authorizeRole(["admin", "user"]), async (req: Request, res, next) => {
  try {
    await supplierController.getSuppliers(req, res);
  } catch (error) {
    next(error);
  }
});
adminRouter.get(
  "/suppliers/:id",authenticateUser,
 authorizeRole(["admin", "user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await supplierController.getSupplierById(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.put(
  "/suppliers/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await supplierController.updateSupplier(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.delete(
  "/suppliers/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => supplierController.deleteSupplier(req, res)
);
// ==============================================
// TRANSFER MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createTransfer",
  authenticateUser,
  authorizeRole(["admin"]),
  transferController.createTransfer
);
adminRouter.get("/transfers",authenticateUser,
 authorizeRole(["admin", "user", "agent"]), transferController.getTransfers);
adminRouter.get("/getTransfer/:id",authenticateUser,
  authorizeRole(["admin"]), transferController.getTransferById);
adminRouter.put(
  "/updateTransfer/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  transferController.updateTransfer
);
adminRouter.delete(
  "/deleteTransfer/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  transferController.deleteTransfer
);
// ==============================================
// ATTRACTION MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/attractions",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req, res, next) => {
    try {
      await attractionController.createAttraction(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.get("/attractions",authenticateUser,
  authorizeRole(["admin", "user", "agent"]), attractionController.getAttractions);
adminRouter.get("/attractions/:id",authenticateUser,
authorizeRole(["admin", "user"]), attractionController.getAttractionById);
adminRouter.put(
  "/attractions/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  attractionController.updateAttraction
);
adminRouter.delete(
  "/attractions/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  attractionController.deleteAttraction
);
// ==============================================
// USER MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createuser",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => userController.createUser(req, res)
);
adminRouter.get("/users",authenticateUser,
  authorizeRole(["admin", "user","agent"]), (req, res) => userController.getUsers(req, res));
adminRouter.get("/users/:id",authenticateUser,
  authorizeRole(["admin"]), (req, res) =>
  userController.getUserById(req, res)
);
adminRouter.get("/users/email/:email", authenticateUser,
  authorizeRole(["admin", "user"]),(req, res) =>
  userController.getUserByEmail(req, res)
);
adminRouter.put(
  "/updateusers/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => userController.updateUser(req, res)
);
adminRouter.delete(
  "/deleteuser/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => userController.deleteUser(req, res)
);
adminRouter.put(
  "/restore/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req, res, next) => {
    try {
      await userController.restore(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.delete(
  "/permanentdelete/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req, res, next) => {
    try {
      await userController.deleteUserPermanently(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.get("/users/team/:id"
  ,authenticateUser,
  authorizeRole(["admin", "user", "agent"]),(req, res) => userController.getUsersByTeamId(req, res));   
// Team management routes
adminRouter.get("/teams",
   async (req, res, next) => {
  try {
    const teams = await teamRepository.getTeams();
    res.json(teams);
  } catch (error) {
    next(error);
  }
});
adminRouter.delete(
  "/teams/:teamId",

  async (req, res, next) => {
    try {
      const { teamId } = req.params;
      await teamRepository.deleteTeamById(teamId);
      res.json({ success: true, message: `Team with ID '${teamId}' deleted.` });
    } catch (error) {
      next(error);
    }
  }
);


// ==============================================
// RESTAURANT MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createrestaurant",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => restaurantController.createRestaurant(req, res)
);
adminRouter.get("/restaurants",authenticateUser,
 authorizeRole(["admin", "user", "agent"]), (req, res) =>
  restaurantController.getRestaurants(req, res)
);
adminRouter.get("/getrestaurant/:id",authenticateUser,authorizeRole(["admin", "user"]), async (req, res, next) => {
  try {
    await restaurantController.getRestaurantById(req, res);
  } catch (error) {
    next(error);
  }
});
adminRouter.put(
  "/updaterestaurant/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => restaurantController.updateRestaurant(req, res)
);
adminRouter.delete(
  "/deleterestaurant/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res) => restaurantController.deleteRestaurant(req, res)
);
// ==============================================
// LEAD MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/leads",
  authenticateUser,
  authorizeRole(["admin"]),
  (req, res, next) => {
    try {
      leadController.createLead(req, res, next);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.get("/leads", (req, res, next) => {
  leadController.getLeads(req, res, next);
});
adminRouter.get(
  "/leads/:id",authenticateUser,
authorizeRole(["admin", "user", "agent"]),
  (req: Request, res: Response, next: NextFunction) => {
    leadController.getLeadById(req, res, next);
  }
);
adminRouter.put(
  "/leads/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) => {
    leadController.updateLead(req, res, next);
  }
);
adminRouter.put(
  "/leads/leadStatus/:id", // ✅ CORRECTED ROUTE
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response) => {
    leadController.updateStatus(req, res);
  }
);
adminRouter.put(
  "/leads/leadStage/:id", // ✅ CORRECTED ROUTE
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response) => {
    leadController.updateStage(req, res);
  }
);
adminRouter.delete(
  "/leads/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) => {
    leadController.deleteLead(req, res, next);
  }
);
adminRouter.get(
  "/leadsbytimeframe/:id",
  (req: Request, res: Response, next: NextFunction) => {
    console.log("Route hit, params:", req.params);
    leadController.getLeadsByTimeframe(req, res, next);
  }
);

adminRouter.post('/assign-to-operations', async (req, res, next) => {
  try {
    await leadController.assignLeadToOperations(req, res);
  } catch (error) {
    next(error);
  }
});

adminRouter.post('/assign-to-assessment', async (req, res, next) => {
  try {
    await leadController.assignLeadToAssessment(req, res);
  } catch (error) {
    next(error);
  }
});

adminRouter.get('/quotation-accepted', (req, res) => {
console.log("route found")
    leadController.getQuotationAcceptedLeads(req, res);
 
  
});
// ==============================================
// PACKAGE MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/packages",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.createPackage(req, res, next)
);
adminRouter.post(
  "/upload-image",
  authenticateUser,
  authorizeRole(["admin"]),
  packageUpload.single("image"),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.uploadImage(req, res, next)
);
adminRouter.put(
  "/packages/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.updatePackage(req, res, next)
);
adminRouter.get(
  "/packages",authenticateUser,
authorizeRole(["admin", "user","agent"]),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.getPackages(req, res, next)
);
adminRouter.get(
  "/packages/:id",authenticateUser,
  authorizeRole(["admin", "user", "agent"]),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.getPackageById(req, res, next)
);
adminRouter.delete(
  "/packages/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  (req: Request, res: Response, next: NextFunction) =>
    packageController.deletePackage(req, res, next)
);
// ==============================================
// QUOTATION MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createQuotations",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req, res, next) => {
    try {
      await quotationController.createQuotation(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.get("/quotations", authenticateUser,
  authorizeRole(["admin", "user", "agent"]), async (req, res) => {
  try {
    await quotationController.getQuotations(req, res);
  } catch (error) {
    console.log(error);
  }
});
adminRouter.get("/quotations/:id", authenticateUser,
 authorizeRole(["admin", "user", "agent"]), async (req, res) => {
  try {
    await quotationController.getQuotationById(req, res);
  } catch (error) {
    console.log(error);
  }
});
adminRouter.get('/quotation/:leadRefNo', authenticateUser,
  authorizeRole(["admin", "user", "agent"]), async (req, res, next) => {
    try {
      await quotationController.getQuotationByLeadRefNo(req, res);
    } catch (error) {
      next(error);
    }
  });
adminRouter.put(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["admin"]),
  async (req, res, next) => {
    try {
      await quotationController.updateQuotation(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.delete(
  "/quotations/:id",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res, next) => {
    try {
      await quotationController.deleteQuotation(req, res);
    } catch (error) {
      next(error);
    }
  }
);

adminRouter.patch(
  "/quotationStatus/:id", authenticateUser,
  authorizeRole(["admin","user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await quotationController.updateQuotationStatus (req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.get(
  "/acceptedQuotation/:leadId", authenticateUser,
  authorizeRole(["admin"]),  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await quotationController.getAcceptedQuotationsByLeadId (req, res);
    } catch (error) {
      next(error);
    }
  }
); 

// DOCUMENT ROUTES
adminRouter.post(
    "/leads/:leadId/documents",
    authenticateUser,
    authorizeRole(["admin", "agent", "user"]),
    documentUpload.single("document"), // Use documentUpload middleware
    async (req: Request, res: Response, next: NextFunction) => {
      try {
        await documentController.uploadDocument(req, res, next);
      } catch (error) {
        next(error);
      }
    }
  );

  adminRouter.get(
    "/leads/:leadId/documents",
    authenticateUser,
    authorizeRole(["admin", "agent","user"]),
    async (req: Request, res: Response, next: NextFunction) => {
      try {
        await documentController.getDocumentsForLead(req, res, next);
      } catch (error) {
        next(error);
      }
    }
  );

  adminRouter.delete(
    "/documents/:id",
    authenticateUser,
    authorizeRole(["admin", "agent", "user"]),
    async (req: Request, res: Response, next: NextFunction) => {
      try {
        await documentController.deleteDocument(req, res, next);
      } catch (error) {
        next(error);
      }
    }
  );

  adminRouter.post("/duplicateQuotation/:id", async (req: Request, res: Response, next: NextFunction) => {
  try {
    await quotationController.duplicateQuotation(req, res);
  } catch (error) {
    next(error);
  }
});

// ==============================================
adminRouter.get("/notification", authenticateUser,
  authorizeRole(["admin"]), getAdminNotifications);

adminRouter.put(
  "/markNotificationAsRead/:id",
  authenticateUser,
  authorizeRole(["admin","user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await markNotificationAsRead(req, res);
    } catch (error) {
      next(error);
    }
  }
);
adminRouter.put(
  "/markAllNotificationAsRead",
  authenticateUser,
  authorizeRole(["admin","user"]),
  async (req: Request, res: Response, next: NextFunction) => {
    try {
      await markAllNotificationsAsRead(req, res);
    } catch (error) {
      next(error);
    }
  }
);
// ==============================================
// GUIDE MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
    "/createGuide",
    authenticateUser,
    authorizeRole(["admin"]),
    guideController.createGuide
);
adminRouter.get("/guides", authenticateUser,
  authorizeRole(["admin", "user", "agent"]), guideController.getGuides);
adminRouter.get("/getGuide/:id", authenticateUser,
  authorizeRole(["admin", "user"]), guideController.getGuideById);
adminRouter.put(
    "/updateGuide/:id",
    authenticateUser,
    authorizeRole(["admin"]),
    guideController.updateGuide
);
adminRouter.delete(
    "/deleteGuide/:id",
    authenticateUser,
    authorizeRole(["admin"]),
    guideController.deleteGuide
);



// ==============================================
// TOUR MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
    "/tours",
    authenticateUser,
    authorizeRole(["admin","user"]),
    (req: Request, res: Response, next: NextFunction) => {
        tourController.createTour(req, res, next).catch(next);  // Handle async errors
    }
);
adminRouter.get(
    "/tours", authenticateUser,
  authorizeRole(["admin", "user", "agent"]),
    (req: Request, res: Response, next: NextFunction) => {
        tourController.getTours(req, res, next).catch(next);
    }
);
adminRouter.get(
    "/tours/:id", authenticateUser,
  authorizeRole(["admin", "user"]),
    (req: Request, res: Response, next: NextFunction) => {
        tourController.getTourById(req, res, next).catch(next);
    }
);
adminRouter.put(
    "/tours/:id",
    authenticateUser,
    authorizeRole(["admin"]),
    (req: Request, res: Response, next: NextFunction) => {
        tourController.updateTour(req, res, next).catch(next);
    }
);
adminRouter.delete(
    "/tours/:id",
    authenticateUser,
    authorizeRole(["admin"]),
    (req: Request, res: Response, next: NextFunction) => {
        tourController.deleteTour(req, res, next).catch(next);
    }
);

export default adminRouter;


// ==============================================
// BOOKING MANAGEMENT ROUTES
// ==============================================
adminRouter.post(
  "/createBooking",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res, next) => {
    try {
      await bookingController.createBooking(req, res);
    } catch (error) {
      next(error);
    }
  }
);

adminRouter.get("/bookings", authenticateUser,
 authorizeRole(["admin", "user"]), async (req, res) => {
  try {
    await bookingController.getBookings(req, res);
  } catch (error) {
    console.log(error);
  }
});

adminRouter.get(
  "/bookings/:id",
  authenticateUser,
  authorizeRole(["admin", "user"]),
  async (req, res, next) => {
    try {
      await bookingController.getBookingById(req, res);
    } catch (error) {
      next(error);
    }
  }
);

adminRouter.patch(
    "/bookings/:id/status",
    authenticateUser,
  authorizeRole(["admin", "user"]),
    (req: Request, res: Response) => {
        bookingController.updateBookingStatus(req, res);
    }
);

adminRouter.patch(
    "/bookings/:id/stage",
    authenticateUser,
    authorizeRole(["admin","user"]),
    (req: Request, res: Response) => {
        bookingController.updateBookingStage(req, res);
    }
);
adminRouter.patch(
    "/bookings/updateAccommodation/:bookingId/:accommodationId",
    authenticateUser,
    authorizeRole(["admin","user"]),
    (req: Request, res: Response) => {
        bookingController.updateAccommodationHandler(req, res);
    }
  );

adminRouter.put(
    '/bookings/updateItinerary/:bookingId/:itineraryId', authenticateUser,
authorizeRole(["admin", "user"]),
   
    (req: Request, res: Response) => {
        bookingController.updateItinerary(req, res);
    }
);

//----------------------------------------------
//final itinerary //
//----------------------------------------------

adminRouter.get(
  "/bookings/:bookingId/final-itinerary",
 authenticateUser,
 authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => bookingController.getFinalItineraryItems(req, res)
);

adminRouter.post(
  "/bookings/:bookingId/final-itinerary/generate-initial", authenticateUser,
authorizeRole(["admin", "user"]),
  // authenticateUser, authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => bookingController.generateInitialFinalItinerary(req, res)
);

adminRouter.post(
  "/bookings/:bookingId/final-itinerary", authenticateUser,
 authorizeRole(["admin", "user"]),
  // authenticateUser, authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => bookingController.createFinalItineraryItem(req, res)
);

adminRouter.put(
  "/bookings/final-itinerary/:itemId",  authenticateUser,
authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => bookingController.updateFinalItineraryItem(req, res)
);

adminRouter.delete(
  "/bookings/final-itinerary/:itemId",
  authenticateUser, authorizeRole(["admin", "user"]),
  (req: Request, res: Response) => bookingController.deleteFinalItineraryItem(req, res)
);