import bcrypt from "bcryptjs";
import prisma from "../../config/prismaClient";

export class UserRepository {
  // In UserRepository
  async createUser(data: any) {
    // Check for existing unique fields
    const existingUser = await this.checkUniqueFields(data);
    if (existingUser) {
      throw this.createConflictError(existingUser, data);
    }

    // Hash password if provided
    if (data.password) {
      data.password = await bcrypt.hash(data.password, 10);
    }

    // Conditionally include agentPrivilege in the data
    const userData: any = {
      name: data.name,
      email: data.email,
      mobile: data.mobile,
      empCode: data.empCode,
      empId: data.empId,
      teamId: data.teamId,
      agentPrivilege: data.agentPrivilege,
      inactive: data.inactive,
    };

    return await prisma.user.create({
      data: userData,
      include: {
        team: true,
      },
    });
  }

  async updateUser(id: string, data: any) {
    // Check for existing unique fields (excluding current user)
    const existingUser = await this.checkUniqueFields(data, id);
    if (existingUser) {
      throw this.createConflictError(existingUser, data);
    }

    const userData: any = {
      name: data.name,
      email: data.email,
      mobile: data.mobile,
      empCode: data.empCode,
      empId: data.empId,
      teamId: data.teamId,
      inactive: data.inactive,
      agentPrivilege: data.agentPrivilege,
    };

    // Only hash the password if it was actually changed
    if (data.password && data.passwordChanged) {
      data.password = await bcrypt.hash(data.password, 10);
      userData.password = data.password; // Add the hashed password to userData
    }

    // If a password is provided BUT it wasn't changed, keep the original password
    else if (data.password) {
      const originalUser = await this.getUserById(id);
      userData.password = originalUser?.password; // Use the original password
    }

    return await prisma.user.update({
      where: { id },
      data: userData,
      include: {
        team: true,
      },
    });
  }

  private async checkUniqueFields(data: any, excludeUserId?: string) {
    const conditions = [];

    if (data.email) {
      conditions.push({ email: data.email });
    }
    if (data.mobile) {
      conditions.push({ mobile: data.mobile });
    }
    if (data.empCode) {
      conditions.push({ empCode: data.empCode });
    }
    if (data.empId) {
      conditions.push({ empId: data.empId });
    }

    if (conditions.length === 0) return null;

    const whereClause = {
      OR: conditions,
      ...(excludeUserId && { NOT: { id: excludeUserId } }),
    };

    return await prisma.user.findFirst({
      where: whereClause,
    });
  }

  private createConflictError(existingUser: any, newData: any) {
    const conflicts = [];

    if (existingUser.email === newData.email) conflicts.push("email");
    if (existingUser.mobile === newData.mobile) conflicts.push("mobile");
    if (existingUser.empCode === newData.empCode)
      conflicts.push("employee code");
    if (existingUser.empId === newData.empId) conflicts.push("employee ID");

    return new Error(
      `User with the same ${conflicts.join(", ")} already exists`
    );
  }

  async getUsers() {
    return await prisma.user.findMany({
      include: {
        team: true,
      },
    });
  }

  async getUserById(id: string) {
    return await prisma.user.findUnique({
      where: { id },
      include: {
        team: true,
      },
    });
  }

  async getUserByEmail(email: string) {
    return await prisma.user.findUnique({
      where: { email },
      include: {
        team: true,
      },
    });
  }

  async deleteUser(id: string) {
    return await prisma.user.update({
      where: { id },
      data: {
        inactive: true,
      },
      include: {
        team: true,
      },
    });
  }

  async restore(id: string) {
    return await prisma.user.update({
      where: { id },
      data: {
        inactive: false,
      },
      include: {
        team: true,
      },
    });
  }

  async deleteUserPermanently(id: string) {
    return await prisma.user.delete({
      where: { id },
    });
  }

  async getInactiveUsers() {
    return await prisma.user.findMany({
      where: {
        inactive: true,
      },
      include: {
        team: true,
      },
    });
  }

  async getUsersByTeamId(teamId: string) {
    return await prisma.user.findMany({
      where: {
        teamId: teamId,
      },
      include: {
        team: true,
      },
    });
  }

  async searchUsers(searchTerm: string) {
    return await prisma.user.findMany({
      where: {
        OR: [
          {
            name: {
              contains: searchTerm,
              mode: "insensitive",
            },
          },
          {
            email: {
              contains: searchTerm,
              mode: "insensitive",
            },
          },
        ],
      },
      include: {
        team: true,
      },
    });
  }
}