import { NotificationService } from "../../services/common/notificationService";
import prisma from "../../config/prismaClient";
import { Lead } from "../../interfaces/Lead";
import { log } from "console";
import { LeadStage, LeadStatus } from "@prisma/client";

export class UserLeadRepository {

  private notificationService: NotificationService;

  constructor() {
    this.notificationService = new NotificationService(); // ✅ Initialize once
  }

  private formatLeadDates(lead: any): Lead {
    return {
      ...lead,
      travelDateFrom: lead.travelDateFrom
        ? new Date(lead.travelDateFrom).toISOString()
        : null,
      travelDateTo: lead.travelDateTo
        ? new Date(lead.travelDateTo).toISOString()
        : null,
    };
  }
 
  public async createLead(leadData: Lead): Promise<Lead> {
    try {

      if (!leadData.empCode) {
        throw new Error('Employee code is missing or invalid');
      }
      console.log("Creating lead with data:", leadData.empCode);
      
      const user = await prisma.user.findUnique({
        where: { empCode: leadData.empCode },
        select: { name: true },
      });
console.log("User found:", user);

  const userName = user?.name || "Unknown User";

      const newLead = await prisma.lead.create({
        data: {
          ...leadData,
          followUps: {
            create: {
              oldStage: 'new_lead',
              newStage: 'new_lead',
              notes: 'Lead created',
              userName: userName, // or get from context
            }
          }
        },
        include: {
          followUps: true,
        }
      });
 // 📦 Save notification to DB
   
   await this.notificationService.notifyAdmin({  // ✅ Use NotificationService instead of direct prisma/io
    type: "lead_created",
    title: "New Lead Created",
    message: `A new lead was created by the user ${userName}`,
    entityType: "lead",
    entityId: newLead.id,
   
  });

      return this.formatLeadDates(newLead);
    } catch (error) {
      console.error("Error creating lead in user lead repository:", error);
      throw error;
    }
  }

  public async getUserLeadsByUserId(userId: string): Promise<Lead[]> {
    try {
      // Step 1: Fetch empCode using userId
      const user = await prisma.user.findUnique({
        where: { id: userId },
        select: { empCode: true },
      });

      if (!user || !user.empCode) {
        throw new Error("User not found or empCode is missing");
      }

      // Step 2: Fetch leads using empCode
      const leads = await prisma.lead.findMany({
        where: {
          empCode: user.empCode,
        },
        include:{
          followUps:true,
          Agent:true,
          User:true
        }
      });

      return leads.map(this.formatLeadDates);
    } catch (error) {
      console.error("Error getting user leads in repository:", error);
      throw error;
    }
  }

  public async getLeadById(leadId: string): Promise<Lead | null> {
    try {
      const lead = await prisma.lead.findUnique({
        where: {
          id: leadId,
        },
        include:{
          followUps:true,
          Agent:true,
          User:true
        }
      });
      return lead ? this.formatLeadDates(lead) : null;
    } catch (error) {
      console.error("Error getting lead by ID in user lead repository:", error);
      throw error;
    }
  }

  public async updateLead(
    leadId: string,
    leadData: Partial<Lead>
  ): Promise<Lead | null> {
    try {
      // 🔍 Step 1: Find the lead by ID
      const lead = await prisma.lead.findUnique({
        where: { id: leadId },
      });
      console.log("Lead found:", lead);

      // ✅ Step 3: Proceed to update
      const updatedLead = await prisma.lead.update({
        where: { id: leadId },
        data: leadData,
      });

      return updatedLead ? this.formatLeadDates(updatedLead) : null;
    } catch (error) {
      console.error("Error updating lead in user lead repository:", error);
      return null;
    }
  }

  public async deleteLead(leadId: string): Promise<void> {
    try {
      await prisma.lead.delete({
        where: {
          id: leadId,
        },
      });
    } catch (error) {
      console.error("Error deleting lead in user lead repository:", error);
      throw error;
    }
  }

  public async updateLeadStatus(id: string, status: LeadStatus): Promise<Lead | null> {
    try {
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: { status },
        include:{
          followUps:true
        }
      });
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead status:", error);
      return null;
    }
  }
  
  public async updateLeadStage(
    id: string,
    newStage: LeadStage,
    userName?: string,
    notes?: string
  ): Promise<Lead | null> {
    try {
      // Get existing lead to fetch oldStage
      const existingLead = await prisma.lead.findUnique({
        where: { id },
        select: { stage: true },
      });
  
      if (!existingLead) return null;
  
      // Optional: Validate userId exists in DB to avoid FK violation
    
  
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: {
          stage: newStage,
          followUps: {
            create: {
              oldStage: existingLead.stage,
              newStage: newStage,
              userName: userName,
              notes: notes || '',
            },
          },
        },
        include: { followUps: true },
      });
  console.log(updatedLead)
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead stage:", error);
      return null;
    }
  }

}