    // backend/src/repositories/UserbookingRepository.ts
    import { Booking, BookingPriceDetailItinerary, BookingPriceDetailsAccommodation } from "@prisma/client";
    import prisma from "../../config/prismaClient";

    export class BookingRepository {
    async fetchQuotationWithDetails(quotationId: string) {
        try {
            return await prisma.quotation.findUnique({
                where: { id: quotationId },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                },
            });
        } catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }

    async handleCreateBooking(quotation: any) {
        try {
            console.log("Quotation in BookingRepository:", quotation);
            if (!quotation) throw new Error("Quotation not found");

            const booking:Booking = await prisma.booking.create({
                data: {
                    reference: `BK-${quotation.refNo}`,
                    leadNo: quotation.leadRefNo,
                    quotationId: quotation.id,
                    travelDates: quotation.travelDates,
                    agent: quotation.leadAgent,
                    quotePrice: Number(quotation.totalAfterMarkup || 0),
                    actualCost: quotation.totalBeforeMarkup || 0,
                    actualPrice: 0,
                    profit: 0,

                    accommodations: {
                        create: (quotation.accommodations ?? []).map((acc: any) => ({
                            hotel: acc.hotel ?? null,
                            room: acc.room ?? null,
                            travelDate: acc.travelDate ? new Date(acc.travelDate) : null,
                            endDate: acc.endDate ? new Date(acc.endDate) : null,
                            nights: acc.nights ?? null,
                            surchargeNights: acc.surchargeNights ?? null,
                            quantity: acc.quantity ?? null,
                            totalPrice: acc.totalPrice ?? null,
                            totalPriceAfterMarkup: acc.totalPriceAfterMarkup ?? null,
                            quotationId: quotation.id,
                            priceDetailsAccommodation: {
                                create: (acc.priceDetailsAccommodation ?? []).map((pd: any) => ({
                                    roomId: pd.roomId ?? null,
                                    pax: pd.pax ?? null,
                                    price: pd.price ?? null,
                                    priceAfterMarkup: pd.priceAfterMarkup ?? null,
                                    costPrice: pd.costPrice ?? null,
                                    optionalPrice: pd.optionalPrice ?? null,
                                    quantity: pd.quantity ?? null,
                                    totalPrice: pd.totalPrice ?? null,
                                    totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
                                })),
                            },
                        })),
                    },

                    itineraries: {
                        create: (quotation.itineraries ?? []).map((itin: any) => ({
                            quotationId: quotation.id,
                            date: itin.date ? new Date(itin.date) : null,
                            label: itin.label ?? null,
                            category: itin.category ?? null,
                            guideId: itin.guideId ?? null,
                            priceDetails: {}, // optionally populate if needed
                            priceDetailsList: {
                                create: (itin.priceDetailsList ?? []).map((pd: any) => ({
                                    category: pd.category ?? null,
                                    price: pd.price ?? null,
                                    priceAfterMarkup: pd.priceAfterMarkup ?? null,
                                    quantity: pd.quantity ?? null,
                                    totalPrice: pd.totalPrice ?? null,
                                    totalPriceAfterMarkup: pd.totalPriceAfterMarkup ?? null,
                                })),
                            },
                        })),
                    },
                },
                include: {
                    lead: true, // ✅ This includes the Lead details
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                },
            });

            console.log("Booking created successfully with lead details:", booking);
            return booking;
        } catch (error) {
            console.error("Error creating booking:", error);
            throw new Error("Failed to create booking from quotation");
        }
    }

    async getAllBookings(userId?: string) {
        try {
            const whereClause: any = {};

            if (userId) {
                whereClause.lead = {
                    assignedOperationId: userId,
                };
            }

            return await prisma.booking.findMany({
                where: whereClause,
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                },
                orderBy: {
                    updatedAt: 'desc',
                },
            });
        } catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve bookings");
        }
    }

    async getBookingById(id: string) {
        try {
            const booking = await prisma.booking.findUnique({
                where: { id },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                        },
                    },
                },
            });

            if (!booking) {
                throw new Error("Booking not found");
            }
            console.log("Booking in repo", booking);
            return booking;
        } catch (error) {
            console.error("Error fetching Booking:", error);
            throw error; // Re-throw to handle in the calling function
        }
    }

    async findBookingWithAccommodations(bookingId: string) {
    return prisma.booking.findUnique({
      where: { id: bookingId },
      include: { accommodations: true },
    });
  }

 async updateAccommodation(
    accommodationId: string,
    updateData: {
        hotel?: string;
        room?: string;
        supplierName?: string;
        supplierContact?: string;
        travelDate?: string;
        nights?: number;
        quotePrice?: number | null;
        actualPrice?: number | null;
        confirmationNo?: string;
        details?: string;
        quantity?: number;
        priceDetailsAccommodation?: BookingPriceDetailsAccommodation[];
    }
) {
    try {
        const existing = await prisma.bookingAccommodation.findUnique({
            where: { id: accommodationId },
        });

        if (!existing) {
            throw new Error("Accommodation not found");
        }

        // Update main accommodation data
        const updatedAccommodation = await prisma.bookingAccommodation.update({
            where: { id: accommodationId },
            data: {
                hotel: updateData.hotel,
                room: updateData.room,
                supplierName: updateData.supplierName,
                supplierContact: updateData.supplierContact,
                travelDate: updateData.travelDate ? new Date(updateData.travelDate) : undefined,
                nights: updateData.nights,
                quotePrice: updateData.quotePrice,
                actualPrice: updateData.actualPrice,
                confirmationNo: updateData.confirmationNo,
                details: updateData.details,
                quantity: updateData.quantity,
            },
        });

        // Handle price details if provided
        if (updateData.priceDetailsAccommodation) {
            // First delete existing price details
            await prisma.bookingPriceDetailsAccommodation.deleteMany({
                where: { accommodationId: accommodationId }
            });

            // Create new price details
            await prisma.bookingPriceDetailsAccommodation.createMany({
                data: updateData.priceDetailsAccommodation.map(detail => ({
                    ...detail,
                    accommodationId: accommodationId,
                    id: undefined // Let database generate ID
                }))
            });
        }

        return updatedAccommodation;
    } catch (error) {
        console.error("Error updating accommodation:", error);
        throw error;
    }
}
async updateItinerary(
    itineraryId: string,
    updateData: {
        date?: string;
        label?: string;
        category?: string;
        supplierId?: string;
        supplierName?: string;
        supplierContact?: string;
        quotePrice?: number | null;
        actualPrice?: number | null;
        confirmationNo?: string;
        details?: string;
        quantity?: number;
        priceDetailsList?: BookingPriceDetailItinerary[];
    }
) {
    try {
        const existing = await prisma.bookingItinerary.findUnique({
            where: { id: itineraryId },
            include: { priceDetailsList: true }
        });

        if (!existing) {
            throw new Error("Itinerary not found");
        }

        // Update main itinerary data
        const updatedItinerary = await prisma.bookingItinerary.update({
            where: { id: itineraryId },
            data: {
                date: updateData.date ? new Date(updateData.date) : undefined,
                label: updateData.label,
                category: updateData.category,
                supplierId: updateData.supplierId,
                supplierName: updateData.supplierName,
                supplierContact: updateData.supplierContact,
                quotePrice: updateData.quotePrice,
                actualPrice: updateData.actualPrice,
                confirmationNo: updateData.confirmationNo,
                details: updateData.details,
                
            },
        });

        // Handle price details if provided
        if (updateData.priceDetailsList) {
            // First delete existing price details
            await prisma.bookingPriceDetailItinerary.deleteMany({
                where: { itineraryId: itineraryId }
            });

            // Create new price details
            await prisma.bookingPriceDetailItinerary.createMany({
                data: updateData.priceDetailsList.map(detail => ({
                    ...detail,
                    itineraryId: itineraryId,
                    id: undefined // Let database generate ID
                }))
            });
        }

        return updatedItinerary;
    } catch (error) {
        console.error("Error updating itinerary:", error);
        throw error;
    }
}


};