import { nanoid } from "nanoid";
import prisma from "../../config/prismaClient";

// Enum to validate vehicle types
const validVehicles = [
  "MIDNIGHT_SURCHARGE",
  "ALPHARD_VELLFIRE",
  "EIGHTEEN_SEATER",
  "STAREX",
  "TEN_SEATER",
  "SEDAN",
];

export class TransferRepository {
  async createTransfer(data: any) {
    // Validate vehicle type before inserting
    for (const vehicle of data.vehicleList) {
      if (!validVehicles.includes(vehicle.vehicle)) {
        throw new Error(`Invalid vehicle type: ${vehicle.vehicle}`);
      }
    }

    return await prisma.transfer.create({
      data: {
        id: nanoid(24),
        name: data.name,
        from: data.from,
        to: data.to,
        supplierId: data.supplierId, // Corrected supplier reference
        terms: data.terms,
        vehicleList: {
          create: data.vehicleList.map((vehicle: any) => ({
            id: nanoid(24),
            vehicle: vehicle.vehicle,
            pax: vehicle.pax,
            price: vehicle.price,
            optionalPrice: vehicle.optionalPrice,
            minPax: vehicle.minPax,
            maxPax: vehicle.maxPax,
            transferId: data.id, // Corrected reference to match Transfer ID
          })),
        },
      },
      include: { vehicleList: true }, // Corrected field name
    });
  }

  async getTransfers() {
    return await prisma.transfer.findMany({ include: { vehicleList: true,  supplier: true } });
  }

  async getTransferById(id: string) {
    return await prisma.transfer.findUnique({
      where: { id },
      include: { vehicleList: true ,  supplier: true },
    });
  }

  async updateTransfer(id: string, data: any) {
    // Validate vehicle type before updating
    for (const vehicle of data.vehicleList) {
      if (!validVehicles.includes(vehicle.vehicle)) {
        throw new Error(`Invalid vehicle type: ${vehicle.vehicle}`);
      }
    }
    console.log("Data.vehicleList:", data.vehicleList);
    return await prisma.transfer.update({
      where: { id },
      data: {
        name: data.name,
        from: data.from,
        to: data.to,
        supplierId: data.supplierId,
        terms: data.terms,
        vehicleList: {
          upsert: data.vehicleList.map((vehicle: any) => ({
            where: { id: vehicle.id ?? nanoid(24) }, // Use existing ID or generate a new one
            create: {
              id: nanoid(24),
              vehicle: vehicle.vehicle,
              pax: vehicle.pax,
              price: vehicle.price,
              optionalPrice: vehicle.optionalPrice,
              minPax: vehicle.minPax,
              maxPax: vehicle.maxPax,
              
            },
            update: {
              vehicle: vehicle.vehicle,
              pax: vehicle.pax,
              price: vehicle.price,
              optionalPrice: vehicle.optionalPrice,
              minPax: vehicle.minPax,
              maxPax: vehicle.maxPax,
              
            },
          })),
        },
      },
      include: { vehicleList: true }, // Corrected field name
    });
  }

  async deleteTransfer(id: string) {
    return await prisma.transfer.delete({
      where: { id },
      include: { vehicleList: true },
    });
  }
}
