
import cuid from "cuid";
import { RestaurantData } from "interfaces/Restaurant";
import prisma from "../../config/prismaClient";

  
  
  export class RestaurantRepository {
    async createRestaurant(data: RestaurantData) {
      try {
        const restaurant = await prisma.restaurant.create({
          data: {
            id: cuid(),
            name: data.name,
            location: data.location,
            targetMarket: data.targetMarket,
            description: data.description,
            termsAndConditions: data.termsAndConditions,
            supplierId: data.supplierId,
            mealPrices: {
              create: data.mealPrices.map((mp) => ({
                category: mp.category,
                pax: mp.pax,
                costPrice: mp.costPrice,
                optionalPrice: mp.optionalPrice,
              })),
            },
          },
          include: {
            mealPrices: true,
          },
        });
        return restaurant;
      } catch (error) {
        console.error("Error creating restaurant:", error);
        throw new Error("Failed to create restaurant");
      }
    }
  
    async getRestaurants() {
      try {
        return await prisma.restaurant.findMany({
          include: {
            mealPrices: true,
            supplier: true
          },
        });
      } catch (error) {
        console.error("Error getting restaurants:", error);
        throw new Error("Failed to get restaurants");
      }
    }
  
    async getRestaurantById(id: string) {
      try {
        return await prisma.restaurant.findUnique({
          where: { id },
          include: {
            mealPrices: true,
            supplier: true
          },
        });
      } catch (error) {
        console.error(`Error getting restaurant with ID ${id}:`, error);
        throw new Error("Failed to get restaurant");
      }
    }
  
    async updateRestaurant(id: string, data: RestaurantData) {
        try {
          // First, delete existing meal prices
          await prisma.mealPrice.deleteMany({
            where: { restaurantId: id },
          });
      
          // Then, update the restaurant and create new meal prices
          const restaurant = await prisma.restaurant.update({
            where: { id: id },
            data: {
              name: data.name,
              location: data.location,
              targetMarket: data.targetMarket,
              description: data.description,
              termsAndConditions: data.termsAndConditions,
              supplierId: data.supplierId,
              mealPrices: {
                create: data.mealPrices.map((mp) => ({
                  category: mp.category,
                  pax: mp.pax,
                  costPrice: mp.costPrice,
                  optionalPrice: mp.optionalPrice,
                })),
              },
            },
            include: {
              mealPrices: true,
            }, // Moved include property outside of data
          });
      
          return restaurant;
        } catch (error) {
          console.error(`Error updating restaurant with ID ${id}:`, error);
          throw new Error("Failed to update restaurant");
        }
      }
      
  
    async deleteRestaurant(id: string) {
      try {
        return await prisma.restaurant.delete({
          where: { id },
        });
      } catch (error) {
        console.error(`Error deleting restaurant with ID ${id}:`, error);
        throw new Error("Failed to delete restaurant");
      }
    }
  }
