import {
  format,
  startOfDay,
  startOfMonth,
  startOfWeek,
  startOfYear,
} from "date-fns";
import prisma from "../../config/prismaClient"; // Import Prisma client
import { Lead } from "../../interfaces/Lead";
import { LeadStage, LeadStatus } from "@prisma/client";
import { NotificationService } from "../../services/common/notificationService";
import { BookingRepository } from "../../repositories/user/UserbookingRepository";

export class LeadRepository {
   private notificationService: NotificationService;
   private bookingRepository: BookingRepository;

  constructor() {
    this.notificationService = new NotificationService();
    this.bookingRepository = new BookingRepository();
  }
  private formatLeadDates(lead: any): Lead {
    return {
      ...lead,
      travelDateFrom: lead.travelDateFrom
        ? new Date(lead.travelDateFrom).toISOString()
        : null,
      travelDateTo: lead.travelDateTo
        ? new Date(lead.travelDateTo).toISOString()
        : null,
    };
  }

  public async createLead(leadData: Lead,userName:string): Promise<Lead> {
    try {
      console.log("LeadRepository createLead input:", leadData);
  
      const newLead = await prisma.lead.create({
        data: {
          ...leadData,
          agentId: leadData.agentId ?? null, // Ensure agentId is explicitly set to null if undefined
          userId: leadData.userId ?? null, // Ensure userId is explicitly set to null if undefined
          followUps: {
            create: {
              oldStage: 'new_lead',
              newStage: 'new_lead',
              notes: 'Lead created',
              userName: userName, // or get from context
            }
          }
        },
        include: {
          followUps: true,
        }
      });
  
      return this.formatLeadDates(newLead);
    } catch (error) {
      console.error("Error creating lead in repository:", error);
      throw error;
    }
  }

  public async getAllLeads(): Promise<Lead[]> {
    try {
      const leadsWithRelations = await prisma.lead.findMany({
        include: {
          Agent: true,
          User: true,
        },
        orderBy: {
          createdOn: 'desc',
        }
      });
      return leadsWithRelations.map(this.formatLeadDates);
    } catch (error) {
      console.error("Error getting leads in repository:", error);
      throw error;
    }
  }

  public async getLeadById(leadId: string): Promise<Lead | null> {
    try {
      const lead = await prisma.lead.findUnique({
        where: { id: leadId },
        include: { 
          Agent: true, // Include agent details
          User: true,  // Include user details
          followUps:true,
          Quotation:true,
        },
      });
      return lead ? this.formatLeadDates(lead) : null;
    } catch (error) {
      console.error("Error getting lead by ID in repository:", error);
      throw error;
    }
  }

  public async updateLead(
    leadId: string,
    leadData: Partial<Lead>
  ): Promise<Lead | null> {
    try {
      const updatedLead = await prisma.lead.update({
        where: { id: leadId },
        data: {
          empCode: leadData.empCode,
          agentName: leadData.agentName,
          agentCode: leadData.agentCode,
          leadNo: leadData.leadNo,
          travelDateFrom: leadData.travelDateFrom,
          travelDateTo: leadData.travelDateTo,
          totalPax: leadData.totalPax,
          description: leadData.description,
          customerName: leadData.customerName,
          email: leadData.email,
          phone: leadData.phone,
          adults: Number(leadData.adults), // ✅ Convert to number
          kids: Number(leadData.kids), // ✅ Convert to number
          infants: Number(leadData.infants), // ✅ Convert to number
          remarks: leadData.remarks,
          agentId: leadData.agentId,
          userId: leadData.userId, // Ensure userId is included if provided
        },
      });
      console.log("Updated Lead:", updatedLead);
      return updatedLead ? this.formatLeadDates(updatedLead) : null;
    } catch (error) {
      console.error("Error updating lead in repository:", error);
      return null;
    }
  }

  public async deleteLead(leadId: string): Promise<void> {
    try {
      await prisma.lead.delete({
        where: { id: leadId },
      });
    } catch (error) {
      console.error("Error deleting lead in repository:", error);
      throw error;
    }
  }

   public async getLeadsByTimeframe(timeframe: string): Promise<any[]> {
    let startDate: Date;
    let endDate: Date = new Date(); // Default end date is now

    switch (timeframe) {
      case "today":
        startDate = startOfDay(new Date());
        break;
      case "weekly":
        startDate = startOfWeek(new Date(), { weekStartsOn: 1 }); // Monday
        break;
      case "month":
        startDate = startOfMonth(new Date());
        break;
      case "year":
        startDate = startOfYear(new Date());
        break;
      default:
        throw new Error(`Invalid timeframe: ${timeframe}`);
    }

    try {
      const leads = await prisma.lead.findMany({
        where: {
          createdOn: {
            gte: startDate,
            lte: endDate,
          },
        },
        select: {
          createdOn: true,
          stage: true,
          status: true,
        },
      });

      const groupedLeads = this.formatLeadsByTimeframe(leads, timeframe);
      console.log("Grouped Leads:", groupedLeads);

      return groupedLeads;
    } catch (error) {
      console.error("Error fetching leads by timeframe:", error);
      throw error;
    }
  }

  private formatLeadsByTimeframe(
    leads: { createdOn: Date; stage: string; status: string }[],
    timeframe: string
  ): any[] {
    const grouped: {
      [key: string]: {
        day: string;
        leads: number;
        booked: number;
        cancelled: number;
      };
    } = {};

    for (const lead of leads) {
      let groupKey: string;

      if (timeframe === "today" || timeframe === "weekly") {
        groupKey = format(lead.createdOn, "EEEE"); // e.g., "Monday"
      } else if (timeframe === "month") {
        groupKey = `Week ${Math.ceil(lead.createdOn.getDate() / 7)}`; // e.g., "Week 1"
      } else if (timeframe === "year") {
        groupKey = format(lead.createdOn, "MMM"); // e.g., "Jan"
      } else {
        groupKey = format(lead.createdOn, "yyyy-MM-dd");
      }

      if (!grouped[groupKey]) {
        grouped[groupKey] = {
          day: groupKey,
          leads: 0,
          booked: 0,
          cancelled: 0,
        };
      }

      grouped[groupKey].leads++;

      if (lead.stage === "quotation_accepted" || lead.stage === "trip_started" || lead.stage === "trip_completed" || lead.stage === "assigned_to_operations" || lead.stage === "assigned_to_assessment" ) { 
        grouped[groupKey].booked++;
      }

      if (lead.stage === "canceled") {
        grouped[groupKey].cancelled++;
      }
    }

    return Object.values(grouped);
  }
  public async updateLeadStatus(id: string, status: LeadStatus): Promise<Lead | null> {
    try {
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: { status },
      });
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead status:", error);
      return null;
    }
  }
  
  public async updateLeadStage(
    id: string,
    newStage: LeadStage,
    userName?: string,
    notes?: string
  ): Promise<Lead | null> {
    try {
      // Get existing lead to fetch oldStage
      const existingLead = await prisma.lead.findUnique({
        where: { id },
        select: { stage: true },
      });
  
      if (!existingLead) return null;
  
      // Optional: Validate userId exists in DB to avoid FK violation
    
  
      const updatedLead = await prisma.lead.update({
        where: { id },
        data: {
          stage: newStage,
          followUps: {
            create: {
              oldStage: existingLead.stage,
              newStage: newStage,
              userName: userName,
              notes: notes || '',
            },
          },
        },
        include: { followUps: true },
      });
  console.log(updatedLead)
      return this.formatLeadDates(updatedLead);
    } catch (error) {
      console.error("Error updating lead stage:", error);
      return null;
    }
  }
  
 public async assignLeadToOperations(leadId: string, operationUserId: string) {
    try {
      const updatedLead = await prisma.$transaction(async (tx) => {
        // 1. Get the current lead and its stage
        const lead = await tx.lead.findUnique({
          where: { id: leadId },
          select: { stage: true, Booking: true, leadNo:true },  // Select related Booking object
        });

        if (!lead) {
          throw new Error("Lead not found.");
        }

        // 2. Get the operation user details
        const operationUser = await tx.user.findUnique({
          where: { id: operationUserId },
          select: { name: true },
        });

        if (!operationUser) {
          throw new Error("Operation user not found.");
        }

        const operationUserName = operationUser.name;

        // 3. Update the lead with assignedOperationId and new stage
        const updated = await tx.lead.update({
          where: { id: leadId },
          data: {
            assignedOperationId: operationUserId,
            stage: 'assigned_to_operations',
          },
        });

        // 4. Add a follow-up entry
        await tx.followUp.create({
          data: {
            id: crypto.randomUUID(),
            date: new Date(),
            oldStage: lead.stage,
            newStage: 'assigned_to_operations',
            notes: `Assigned to ${operationUserName}, Operation Team`,
            userName: 'Admin',
            leadId: leadId,
          },
        });

        return updated;
      });

      // Find the booking ID related to the lead
      const booking = await prisma.booking.findFirst({
        where: {
          lead: {
            id: leadId,
          },
        },
      });

      const bookingId = booking?.id;

      console.log(bookingId,"booking id from assigned to operation ");
      

      // Send notification to the operations user
      await this.notificationService.notifyUser(operationUserId, {
        type: "lead_assigned_operations",
        title: "Lead Assigned to Operations",
        message: `Lead ${updatedLead.leadNo} has been assigned to you in the Operations team.`, // Use leadNo if available
        entityType: "lead", // Use booking as entity type
        entityId: bookingId || leadId, // Use booking ID if exists, else use lead ID
      });


      return updatedLead;

    } catch (error) {
      console.error('Error assigning lead to operations user:', error);
      throw new Error('Failed to assign lead to operations user.');

    }
  }

public async assignLeadToAssessment(leadId: string, assessmentUserId: string) {
    try {
        // Input Validation: Important for security
        if (!leadId || !assessmentUserId) {
            throw new Error("Lead ID and Assessment User ID are required.");
        }

        const updatedLead = await prisma.$transaction(async (tx) => {
            // 1. Get the current lead and its stage, assignedToId, and assignedAssessmentId
            const lead = await tx.lead.findUnique({
                where: { id: leadId },
                select: {
                    stage: true,
                    assignedToId: true,
                    leadNo: true,
                    assignedAssessmentId: true,
                    assignedAssessmentOn: true,
                },
            });

            if (!lead) {
                throw new Error("Lead not found.");
            }

            // Check if the lead is already assigned to this assessment user
            if (lead.assignedAssessmentId === assessmentUserId) {
                throw new Error("Lead is already assigned to this assessment user.");
            }

            // 2. Get the assessment user details
            const assessmentUser = await tx.user.findUnique({
                where: { id: assessmentUserId },
                select: { name: true },
            });

            if (!assessmentUser) {
                throw new Error("Assessment user not found.");
            }

            const assessmentUserName = assessmentUser.name;

            // 3. Update the lead with assignedAssessmentId and new stage
            const updated = await tx.lead.update({
                where: { id: leadId },
                data: {
                    assignedAssessmentId: assessmentUserId, // Assign the lead to the assessment user
                    stage: 'assigned_to_assessment',
                    assignedAssessmentOn: new Date(),
                },
            });

            // 4. Add a follow-up entry
            await tx.followUp.create({
                data: {
                    id: crypto.randomUUID(),
                    date: new Date(),
                    oldStage: lead.stage,
                    newStage: 'assigned_to_assessment',
                    notes: `Assigned to ${assessmentUserName} for assessment.`,
                    userName: 'Admin',  // Or the current user's name
                    leadId: leadId,
                },
            });

            return updated;
        });

        // Find the booking ID related to the lead
        const booking = await prisma.booking.findFirst({
            where: {
                lead: { id: leadId },
            },
        });

        const bookingId = booking?.id;

        await this.notificationService.notifyUser(assessmentUserId, {
            type: "lead_assigned_assessment",
            title: "Lead Assigned to Assessment",
            message: `Lead ${updatedLead.leadNo} has been assigned to you for assessment. Please review it.`,
            entityType: "assessment",  // Consistent with Operations - using Lead entity type
            entityId: bookingId || leadId,
        });

        return updatedLead;

    } catch (error) {
        console.error('Error assigning lead to assessment:', error);
        throw new Error('Failed to assign lead to assessment.');
    }
}

  
public async findQuotationAcceptedLeads() {
  try {
    console.log("repo called");

    const acceptedLead = await prisma.lead.findMany({
      where: {
        stage: "quotation_accepted",
      },
      include: {
        Agent: true,
        Quotation: {
          where: {
            status: "Accepted",
          },
          include: {
            itineraries: {  
              include: {
                priceDetailsList: true,
              },
            },
            accommodations: {
              include: {
                priceDetailsAccommodation: true,
              },
            },
          },
        },
      },
    });

    console.log(acceptedLead);
    return acceptedLead;

  } catch (error) {
    console.error("Error fetching quotation accepted leads:", error);
    throw new Error("Failed to fetch leads with stage 'quotation_accepted'");
  }
}



  //   public async createManyLeads(leads: Lead[]) {
  //     try {
  //       // Validate and transform data
  //       const validatedLeads = leads
  //         .filter((lead) => !!lead.leadNo) // Ensure leadNo exists
  //         .map((lead) => ({
  //           ...lead,
  //           createdOn: new Date(lead.createdOn || Date.now()),
  //         }));

  //       const result = await prisma.lead.createMany({
  //         data: validatedLeads,
  //         skipDuplicates: true,
  //       });

  //       return result;
  //     } catch (error) {
  //       console.error("Error creating leads:", error);
  //       throw new Error("Failed to create leads.");
  //     }
  //   }
}
