// backend/src/repositories/admin/bookingRepository.ts
import {
  Booking,
  BookingItinerary,
  BookingPriceDetailItinerary,
  BookingPriceDetailsAccommodation,
  FinalItineraryItem,
  Prisma, // Import Prisma for Prisma.BookingGetPayload and other types
  HotelVoucher,
  BookingAccommodation, // Make sure HotelVoucher type is available if used
} from "@prisma/client";
import prisma from "../../config/prismaClient";
import { addDays, format } from 'date-fns';

// Define a type for the booking object with all its includes from getBookingById
type BookingWithFullDetails = Prisma.BookingGetPayload<{
  include: {
    hotelVouchers: true;
    accommodations: {
      include: { priceDetailsAccommodation: true };
    };
    itineraries: {
      include: { priceDetailsList: true };
    };
    lead: {
      include: {
        Agent: true;
      };
    };
  };
}>;
const convertToMYR = (price: number | null | undefined, rate: number): number | null => {
  // If the price is null, undefined, or not a number, return null to avoid DB errors.
  if (price === null || price === undefined || isNaN(Number(price))) {
    return null;
  }
  // The rate is how many MYR one unit of the foreign currency is worth.
  // So, we multiply the foreign currency price by the rate to get the MYR value.
  return Number(price) * rate;
};
// Type for the updateData in updateAccommodation for clarity
type BookingAccommodationUpdateData = {
  hotel?: string;
  room?: string;
  supplierName?: string;
  supplierContact?: string;
  travelDate?: string; // Will be converted to Date
  nights?: number;
  quotePrice?: number | null;
  actualPrice?: number | null;
  confirmationNo?: string; // Note: Prisma model uses 'confirmationNo'
  details?: string;
  quantity?: number;
  // This should be the Prisma generated type for create/update input
  priceDetailsAccommodation?: Prisma.BookingPriceDetailsAccommodationCreateManyAccommodationInput[];
};


export class BookingRepository {
  async fetchQuotationWithDetails(quotationId: string) {
    try {
      return await prisma.quotation.findUnique({
        where: { id: quotationId },
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          lead: true,
          Agent: true, // Include agent details if needed
        },
      });
    } catch (error) {
      console.error("Error fetching quotation details:", error);
      throw new Error("Failed to fetch quotation details");
    }
  }

 async handleCreateBooking(quotation: any) { // Consider creating a strict Quotation type
  try {
    console.log("Quotation received for booking creation:", quotation);
    if (!quotation) throw new Error("Quotation data is missing.");

    // Determine the conversion rate. If currency is MYR, rate is 1.
    // Otherwise, use the provided rate.
    const conversionRate = (quotation.currency !== 'MYR' && quotation.currencyRate) 
      ? Number(quotation.currencyRate) 
      : 1;

    // Add a safeguard for an invalid rate
    if (isNaN(conversionRate) || conversionRate <= 0) {
      throw new Error(`Invalid currency rate for ${quotation.currency}: ${quotation.currencyRate}`);
    }

    const booking = await prisma.booking.create({
      data: {
        // Booking-level data
        reference: quotation.refNo.replace(/^QT-/, 'BK-'),
        leadNo: quotation.leadRefNo,
        quotationId: quotation.id,
        travelDates: quotation.travelDates,
        agent: quotation.leadAgent ?? null,
        guestName: quotation.guestName ?? null,

        // ** IMPORTANT: Set new booking currency to MYR after conversion **
        currency: 'MYR',
        currencyRate: 1,

        // Convert top-level prices to MYR
        quotePrice: convertToMYR(quotation.totalAfterMarkup, conversionRate) ?? 0,
        actualCost: convertToMYR(quotation.totalBeforeMarkup, conversionRate) ?? 0,
        actualPrice: 0, // Stays 0 initially
        profit: 0,      // Stays 0 initially

        // Nested accommodations with converted prices
        accommodations: {
          create: (quotation.accommodations ?? []).map((acc: any) => ({
            // Non-price fields remain the same
            hotel: acc.hotel ?? null,
            hotelAddress: acc.hotelAddress ?? null,
            supplierId: acc.supplierId ?? null,
            supplierName: acc.supplierName ?? null,
            supplierContact: acc.supplierContact ?? null,
            details: acc.details ?? null,
            confirmationNo: acc.confirmationNo ?? null,
            room: acc.room ?? null,
            travelDate: acc.travelDate ? new Date(acc.travelDate) : null,
            endDate: acc.endDate ? new Date(acc.endDate) : null,
            nights: acc.nights ?? null,
            surchargeNights: acc.surchargeNights ?? null,
            quantity: acc.quantity ?? null,
            quotationId: quotation.id,

            // Convert accommodation-level price to MYR
            quotePrice: convertToMYR(acc.totalPriceAfterMarkup, conversionRate),
            
            priceDetailsAccommodation: {
              create: (acc.priceDetailsAccommodation ?? []).map((pd: any) => ({
                roomId: pd.roomId ?? null,
                category: pd.category ?? null,
                pax: pd.pax ?? null,
                quantity: pd.quantity ?? null,

                // Convert all nested price details to MYR
                price: convertToMYR(pd.price, conversionRate),
                quotePrice: convertToMYR(pd.priceAfterMarkup, conversionRate),
                costPrice: convertToMYR(pd.costPrice, conversionRate),
                optionalPrice: convertToMYR(pd.optionalPrice, conversionRate),
                totalPrice: convertToMYR(pd.totalPrice, conversionRate),
                totalPriceAfterMarkup: convertToMYR(pd.totalPriceAfterMarkup, conversionRate),
              })),
            },
          })),
        },
        
        // Nested itineraries with converted prices
        itineraries: {
          create: (quotation.itineraries ?? []).map((itin: any) => ({
            // Non-price fields remain the same
            quotationId: quotation.id,
            supplierId: itin.supplierId ?? null,
            supplierName: itin.supplierName ?? null,
            supplierContact: itin.supplierContact ?? null,
            details: itin.details ?? null,
            confirmationNo: itin.confirmationNo ?? null,
            date: itin.date ? new Date(itin.date) : null,
            label: itin.label ?? null,
            category: itin.category ?? null,
            guideId: itin.guideId ?? null,
            priceDetails: itin.priceDetails ?? Prisma.JsonNull,

            // Convert itinerary-level price to MYR
            quotePrice: convertToMYR(itin.totalPriceAfterMarkup, conversionRate),
            
            priceDetailsList: {
              create: (itin.priceDetailsList ?? []).map((pd: any) => ({
                category: pd.category ?? null,
                quantity: pd.quantity ?? null,
                
                // Convert all nested price details to MYR
                price: convertToMYR(pd.price, conversionRate),
                quotePrice: convertToMYR(pd.totalPriceAfterMarkup, conversionRate), // As per original structure
                totalPrice: convertToMYR(pd.totalPrice, conversionRate),
                totalPriceAfterMarkup: convertToMYR(pd.totalPriceAfterMarkup, conversionRate),
              })),
            },
          })),
        },
      },
      // Include relations in the returned object
      include: {
        lead: true,
        accommodations: { include: { priceDetailsAccommodation: true } },
        itineraries: { include: { priceDetailsList: true } },
      },
    });

    console.log("Booking created successfully in MYR:", booking);
    return booking;
  } catch (error) {
    console.error("Error creating booking with currency conversion:", error);
    // Be specific about the error source if possible
    if (error instanceof Error) {
      throw new Error(`Failed to create booking from quotation: ${error.message}`);
    } else {
      throw new Error("Failed to create booking from quotation: Unknown error");
    }
  }
}

  async getAllBookings() {
    try {
      return await prisma.booking.findMany({
        include: {
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          lead: true,
        },
        orderBy: {
          updatedAt: 'desc',
        },
      });
    } catch (error) {
      console.error("Error fetching all bookings:", error);
      throw new Error("Failed to retrieve quotations");
    }
  }

  async getBookingById(id: string): Promise<BookingWithFullDetails | null> {
    try {
      const booking = await prisma.booking.findUnique({
        where: { id },
        include: {
          hotelVouchers: true,
          accommodations: {
            include: { priceDetailsAccommodation: true },
          },
          itineraries: {
            include: { priceDetailsList: true },
          },
          lead: {
            include: {
              Agent: true,
            },
          },
        },
      });

      if (!booking) {
        return null;
      }
      return booking;
    } catch (error) {
      console.error("Error fetching Booking:", error);
      throw error;
    }
  }

  async findBookingWithAccommodations(bookingId: string) {
    return prisma.booking.findUnique({
      where: { id: bookingId },
      include: { accommodations: true }, // This will include BookingAccommodation records
    });
  }

  // This is the full updateAccommodation method from your first snippet
 // Assuming necessary imports like `Prisma`, `addDays`, `format`, `HotelVoucher` are already in place.

// ... inside your repository class ...

async updateAccommodation(
  accommodationId: string,
  updateData: BookingAccommodationUpdateData // Using the defined type
) {
  try {
    const updatedAccommodation = await prisma.bookingAccommodation.update({
      where: { id: accommodationId },
      data: {
        hotel: updateData.hotel,
        room: updateData.room,
        supplierName: updateData.supplierName,
        supplierContact: updateData.supplierContact,
        travelDate: updateData.travelDate ? new Date(updateData.travelDate) : undefined,
        nights: updateData.nights,
        quotePrice: updateData.quotePrice,
        actualPrice: updateData.actualPrice,
        confirmationNo: updateData.confirmationNo,
        details: updateData.details,
        quantity: updateData.quantity,
      },
    });

    if (!updatedAccommodation) {
      throw new Error("Accommodation not found or failed to update.");
    }

    // This part remains the same: update price details if they are provided
    if (updateData.priceDetailsAccommodation && updateData.priceDetailsAccommodation.length > 0) {
      await prisma.bookingPriceDetailsAccommodation.deleteMany({
        where: { accommodationId: accommodationId }
      });
      await prisma.bookingPriceDetailsAccommodation.createMany({
        data: updateData.priceDetailsAccommodation.map(detail => ({
          ...detail,
          accommodationId: accommodationId,
        })),
      });
    }

    // --- START: MODIFIED LOGIC FOR VOUCHER DATA ---

    // Re-fetch the accommodation WITH its relations AFTER price details have been updated.
    // This is crucial to get the latest data for voucher creation.
    const accommodationWithRelations = await prisma.bookingAccommodation.findUnique({
      where: { id: accommodationId },
      include: {
        booking: {
          select: {
            id: true,
            reference: true,
            leadNo: true,
            lead: {
              select: {
                customerName: true,
                totalPax: true,
                adults: true,
                kids: true,
              }
            }
          },
        },
        hotelVoucher: true,
        priceDetailsAccommodation: true, // IMPORTANT: Include the price details
      }
    });

    if (!accommodationWithRelations) {
        throw new Error("Failed to re-fetch accommodation after update.");
    }

    // --- NEW VOUCHER PARSING LOGIC ---
    let mealPlanItems: Set<string> = new Set();
    let specialInclusionItems: string[] = [];
    let roomDetailsDescription = `${accommodationWithRelations.priceDetailsAccommodation[0]?.quantity || 1} X ${accommodationWithRelations.room || 'Standard Room'}`;

    // Iterate through the structured price details
    if (accommodationWithRelations.priceDetailsAccommodation) {
      accommodationWithRelations.priceDetailsAccommodation.forEach(detail => {
        if (detail.category) {
          const categoryLower = detail.category.toLowerCase();

          // Check for meal plan keywords
          if (categoryLower.includes('breakfast')) {
            mealPlanItems.add('Breakfast');
          }
          if (categoryLower.includes('child breakfast')) {
            mealPlanItems.add('Child Breakfast');
          }
          // Add more meal plan checks here if needed (e.g., HB, FB)

          // Check for special inclusion keywords
          if (categoryLower.includes('extra bed')) {
            // You can format this string however you like
            specialInclusionItems.push(`${detail.quantity || 1} x Extra Bed (${detail.pax || ''} )`);
          }
          // Add more special inclusion checks here if needed (e.g., honeymoon setup)
        }
      });
    }

    // Also check the general `details` field as a fallback
    if (accommodationWithRelations.details) {
        const detailsLower = accommodationWithRelations.details.toLowerCase();
        if (detailsLower.includes("bb") || detailsLower.includes("breakfast")) mealPlanItems.add("Breakfast");
        else if (detailsLower.includes("hb") || detailsLower.includes("child breakfast")) mealPlanItems.add("Child Breakfast");
        else if (detailsLower.includes("fb") || detailsLower.includes("full board")) mealPlanItems.add("FB");
        else if (detailsLower.includes("ai") || detailsLower.includes("all inclusive")) mealPlanItems.add("AI");
    }
    
    // Finalize the strings
    const finalMealPlan = mealPlanItems.size > 0 ? Array.from(mealPlanItems).join(' & ') : "N/A";
    const finalSpecialInclusions = specialInclusionItems.length > 0 ? specialInclusionItems.join(', ').toUpperCase() : "N/A";

    // Build room description
    if (finalMealPlan !== "N/A") {
      roomDetailsDescription += ` WITH ${finalMealPlan}`;
    }
    roomDetailsDescription += ` X ${accommodationWithRelations.nights || 0} Nights`;
    
    // --- END: NEW VOUCHER PARSING LOGIC ---

    const checkInDate = accommodationWithRelations.travelDate ? new Date(accommodationWithRelations.travelDate) : new Date();
    const nights = accommodationWithRelations.nights || 0;
    const checkOutDateObj = addDays(checkInDate, nights);

    const hotelVoucherDataForDb = {
      customerName: accommodationWithRelations.booking?.lead?.customerName || 'N/A',
      totalAdults: accommodationWithRelations.booking?.lead?.adults || 0,
      totalKids: accommodationWithRelations.booking?.lead?.kids || 0,
      hotelName: accommodationWithRelations.hotel || 'N/A Hotel',
      hotelAddress: accommodationWithRelations.hotelAddress || 'Hotel Address Not Provided',
      hotelPhone: accommodationWithRelations.supplierContact || 'Hotel Phone Not Provided',
      checkInDate: checkInDate,
      checkOutDate: checkOutDateObj,
      roomDetails: roomDetailsDescription,
      mealPlan: finalMealPlan, // Use the newly parsed meal plan
      confirmationNumber: accommodationWithRelations.confirmationNo || 'N/A',
      specialInclusions: finalSpecialInclusions, // Use the newly parsed inclusions
      termsAndConditions: "Standard hotel terms apply. Please refer to hotel policy.",
    };

    // --- The rest of your voucher creation/update logic remains the same ---
    
    console.log("Hotel Voucher data prepared for DB:", hotelVoucherDataForDb);
    let persistedHotelVoucher: HotelVoucher;
    const uniqueVoucherNo = `HV-${accommodationWithRelations.booking?.reference?.replace('BK-','') || 'DEFREF'}-${accommodationWithRelations.id.slice(-4).toUpperCase()}`;
    const bookingIdForVoucher = accommodationWithRelations.bookingId;

    if (accommodationWithRelations.hotelVoucher) {
      persistedHotelVoucher = await prisma.hotelVoucher.update({
        where: { id: accommodationWithRelations.hotelVoucher.id },
        data: {
          ...hotelVoucherDataForDb,
          ...(bookingIdForVoucher && { booking: { connect: { id: bookingIdForVoucher } } }),
        },
      });
      console.log("HotelVoucher updated:", persistedHotelVoucher.id);
    } else {
      persistedHotelVoucher = await prisma.hotelVoucher.create({
        data: {
          ...hotelVoucherDataForDb,
          voucherNo: uniqueVoucherNo,
          accommodation: { connect: { id: accommodationId } },
          ...(bookingIdForVoucher && { booking: { connect: { id: bookingIdForVoucher } } }),
        },
      });
      console.log("HotelVoucher created:", persistedHotelVoucher.id);
    }
    
    // ... your frontendVoucherObject creation and final return ...
     const frontendVoucherObject = {
        id: persistedHotelVoucher.id,
        voucherNumber: persistedHotelVoucher.voucherNo,
        type: 'Hotel' as 'Hotel',
        serviceName: persistedHotelVoucher.hotelName,
        customerName: persistedHotelVoucher.customerName || 'N/A',
        totalAdults: persistedHotelVoucher.totalAdults || 0,
        totalKids: persistedHotelVoucher.totalKids || 0,
        hotelAddress: persistedHotelVoucher.hotelAddress || 'N/A',
        hotelPhone: persistedHotelVoucher.hotelPhone || 'N/A',
        confirmationNumber: persistedHotelVoucher.confirmationNumber || 'N/A',
        checkInDate: format(new Date(persistedHotelVoucher.checkInDate), 'dd.MM.yyyy'),
        checkOutDate: format(new Date(persistedHotelVoucher.checkOutDate), 'dd.MM.yyyy'),
        nights: nights,
        roomDetailsDescription: persistedHotelVoucher.roomDetails,
        mealPlan: persistedHotelVoucher.mealPlan || 'N/A',
        specialInclusions: persistedHotelVoucher.specialInclusions || 'N/A',
        status: 'Confirmed' as 'Confirmed',
      };
      
      return {
        updatedAccommodation: accommodationWithRelations, // Return the object with all relations
        voucher: frontendVoucherObject,
      };

  } catch (error) {
    console.error("Error in updateAccommodationAndManageHotelVoucher:", error);
    if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2002') {
      throw new Error("A voucher with this number already exists.");
    }
    throw error;
  }
}

  async updateItinerary(
    itineraryId: string,
    updateData: {
      date?: string;
      label?: string;
      category?: string;
      supplierId?: string;
      supplierName?: string;
      supplierContact?: string;
      quotePrice?: number | null;
      actualPrice?: number | null;
      confirmationNo?: string; // Changed from confirmationNo to match service/controller
      details?: string;
      priceDetailsList?: Prisma.BookingPriceDetailItineraryCreateManyItineraryInput[];
    }
  ) {
    try {
      const existing = await prisma.bookingItinerary.findUnique({
        where: { id: itineraryId },
        include: { priceDetailsList: true }
      });

      if (!existing) {
        throw new Error("Itinerary not found");
      }

      await prisma.bookingItinerary.update({
        where: { id: itineraryId },
        data: {
          date: updateData.date ? new Date(updateData.date) : undefined,
          label: updateData.label,
          category: updateData.category,
          supplierId: updateData.supplierId,
          supplierName: updateData.supplierName,
          supplierContact: updateData.supplierContact,
          quotePrice: updateData.quotePrice, // Decimal
          actualPrice: updateData.actualPrice, // Decimal
          confirmationNo: updateData.confirmationNo, // Schema uses confirmationNo
          details: updateData.details,
        },
      });

      if (updateData.priceDetailsList) {
        await prisma.bookingPriceDetailItinerary.deleteMany({
          where: { itineraryId: itineraryId }
        });
        await prisma.bookingPriceDetailItinerary.createMany({
          data: updateData.priceDetailsList.map(detail => ({
            ...detail,
            itineraryId: itineraryId,
          }))
        });
      }
      return await prisma.bookingItinerary.findUnique({
        where: { id: itineraryId },
        include: { priceDetailsList: true }
      });
    } catch (error) {
      console.error("Error updating itinerary:", error);
      throw error;
    }
  }

  async getFinalItineraryItemsByBookingId(bookingId: string): Promise<FinalItineraryItem[]> {
    return prisma.finalItineraryItem.findMany({
      where: { bookingId },
      orderBy: [
        { day: 'asc' },
        { time: 'asc' },
      ],
    });
  }

  async createFinalItineraryItem(
    bookingId: string,
    data: Omit<FinalItineraryItem, 'id' | 'bookingId' | 'createdAt' | 'updatedAt' | 'booking'>
  ): Promise<FinalItineraryItem> {
    return prisma.finalItineraryItem.create({
      data: {
        ...data,
        bookingId,
      },
    });
  }

  async updateFinalItineraryItem(
    itemId: string,
    data: Partial<Omit<FinalItineraryItem, 'id' | 'bookingId' | 'createdAt' | 'updatedAt' | 'booking'>>
  ): Promise<FinalItineraryItem> {
    return prisma.finalItineraryItem.update({
      where: { id: itemId },
      data,
    });
  }

  async deleteFinalItineraryItem(itemId: string): Promise<FinalItineraryItem> {
    return prisma.finalItineraryItem.delete({
      where: { id: itemId },
    });
  }

  async generateInitialFinalItineraryForBooking(bookingId: string): Promise<FinalItineraryItem[]> {
    const booking = await prisma.booking.findUnique({
      where: { id: bookingId },
      include: {
        accommodations: true, // This includes BookingAccommodation model which has hotelAddress
        itineraries: true,
      },
    });

    if (!booking) {
      throw new Error('Booking not found');
    }

    const itemsToCreate: Omit<FinalItineraryItem, 'id' | 'bookingId' | 'createdAt' | 'updatedAt' | 'booking'>[] = [];

    const getDayNumber = (itemDate: Date | string | null): number => {
      if (!itemDate || !booking.travelDates) return 1;
      const tripStartDateStr = booking.travelDates.split(" to ")[0];
      const tripStart = new Date(tripStartDateStr);
      const currentItemDate = new Date(itemDate);
      if (isNaN(tripStart.getTime()) || isNaN(currentItemDate.getTime())) return 1;
      const diffTime = Math.abs(currentItemDate.setHours(0, 0, 0, 0) - tripStart.setHours(0, 0, 0, 0));
      const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
      return diffDays + 1;
    };

    // booking.accommodations is an array of BookingAccommodation
    booking.accommodations.forEach(acc => { // acc is BookingAccommodation
      if (acc.hotel && acc.travelDate) {
        itemsToCreate.push({
          day: getDayNumber(acc.travelDate),
          activity: `Check-in: ${acc.hotel}`,
          location: acc.hotelAddress || acc.hotel || null, // acc.hotelAddress should exist here
          time: '14:00',
          sourceType: 'ACCOMMODATION',
          sourceId: acc.id,
          notes: acc.details || null
        });
      }
      if (acc.hotel && acc.endDate) {
        itemsToCreate.push({
          day: getDayNumber(acc.endDate),
          activity: `Check-out: ${acc.hotel}`,
          location: acc.hotelAddress || acc.hotel || null, // acc.hotelAddress should exist here
          time: '12:00',
          sourceType: 'ACCOMMODATION',
          sourceId: acc.id,
          notes: null
        });
      }
    });

    booking.itineraries.forEach(itin => {
      if (itin.label && itin.date) {
        itemsToCreate.push({
          day: getDayNumber(itin.date),
          activity: itin.label,
          location: itin.category || null,
          time: '10:00',
          sourceType: 'BOOKING_ITINERARY',
          sourceId: itin.id,
          notes: itin.details || null
        });
      }
    });

    if (itemsToCreate.length > 0) {
      await prisma.finalItineraryItem.createMany({
        data: itemsToCreate.map(item => ({ ...item, bookingId })),
        skipDuplicates: true,
      });
    }
    return this.getFinalItineraryItemsByBookingId(bookingId);
  }
}