import { nanoid } from "nanoid";
import prisma from "../../config/prismaClient";
import { Attraction } from "../../interfaces/Attraction";

export class AttractionRepository {
    async createAttraction(data: Attraction) {
        try {
            const { prices, surchargeDates, closedDates, supplierId, ...attractionData } = data;

            const createdAttraction = await prisma.attraction.create({
                data: {
                    id: nanoid(24),
                    ...attractionData,
                    surchargeDates: surchargeDates.map(date => new Date(date)),
                    closedDates: closedDates.map(date => new Date(date)),
                    supplier: {
                        connect: {
                            id: supplierId,
                        },
                    },
                    prices: {
                        create: prices.map(price => ({
                            category: price.category,
                            paxType: price.paxType,
                            costPrice: price.costPrice,
                            optionalPrice: price.optionalPrice,
                        })),
                    },
                },
                include: {
                    prices: true,
                },
            });

            return createdAttraction;
        } catch (error: any) {
            console.error("Error creating attraction in repository:", error);
            throw new Error(error.message || "Failed to create attraction");
        }
    }

    async getAttractions() {
        try {
            return await prisma.attraction.findMany({
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
        } catch (error: any) {
            console.error("Error fetching attractions:", error);
            throw new Error(error.message || "Failed to fetch attractions");
        }
    }

    async getAttractionById(id: string) {
        try {
            return await prisma.attraction.findUnique({
                where: { id },
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
        } catch (error: any) {
            console.error(`Error fetching attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to fetch attraction");
        }
    }

    async updateAttraction(id: string, data: Attraction) {
        try {
            const { prices, surchargeDates, closedDates, supplierId, ...attractionData } = data;

            // Delete existing prices
            await prisma.attractionPrice.deleteMany({
                where: {
                    attractionId: id,
                },
            });

            const updatedAttraction = await prisma.attraction.update({
                where: { id },
                data: {
                    ...attractionData,
                    surchargeDates: surchargeDates.map(date => new Date(date)),
                    closedDates: closedDates.map(date => new Date(date)),
                    supplier: {
                        connect: {
                            id: supplierId,
                        },
                    },
                    prices: {
                        create: prices.map(price => ({
                            category: price.category,
                            paxType: price.paxType,
                            costPrice: price.costPrice,
                            optionalPrice: price.optionalPrice,
                        })),
                    },
                },
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
            return updatedAttraction;
        } catch (error: any) {
            console.error(`Error updating attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to update attraction");
        }
    }

    async deleteAttraction(id: string) {
        try {
            await prisma.attraction.delete({
                where: { id },
            });
        } catch (error: any) {
            console.error(`Error deleting attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to delete attraction");
        }
    }
}
