// File: backend/src/controllers/user/documentController.ts

import { Request, Response, NextFunction } from 'express';
import { DocumentService } from '../../services/user/documentService';
import { cloudConfigDocument, uploadToCloudDocument } from '../../config/cloudStorageDocument';
import fs from 'fs';
import path from 'path';
import { Document } from '../../interfaces/Document';
import { nanoid } from 'nanoid';
import { documentUpload } from '../../config/multerConfig'; // Import documentUpload
import { User } from '@prisma/client';
import prisma from '../../config/prismaClient';

//Create a custom Request interface
interface CustomRequest extends Request {
    user?: { id: string; role: string };
}
export class DocumentController {
  private documentService: DocumentService;

  constructor(documentService: DocumentService) {
    this.documentService = documentService;
  }

  public async uploadDocument(req: CustomRequest, res: Response, next: NextFunction): Promise<void> {
      try {
          const leadId = req.params.leadId;

          if (!req.file) {
              res.status(400).json({ message: 'No file uploaded' });
              return;
          }

          // Sanitize filename
          const sanitizedFilename = path.basename(req.file.originalname);

          // Get file extension
          const fileExtension = path.extname(sanitizedFilename).toLowerCase();

          // Generate unique remote filename
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
          const remoteFileName = `document-${uniqueSuffix}${fileExtension}`;

          // Construct local file path
          const localFilePath = req.file.path;

          try {
              // Upload to Filezilla
              await uploadToCloudDocument(localFilePath, remoteFileName);
              const fileUrl = cloudConfigDocument.baseUrl + remoteFileName;

              // Delete the local file after successful upload
              fs.unlink(localFilePath, (err) => {
                  if (err) {
                      console.error('Error deleting local file:', err);
                  } else {
                      console.log('Local file deleted successfully');
                  }
              });

          let uploadedBy: string = 'Unknown User';

          if (req.user?.role === 'user') {
          
          } else {
              try {
                  // Fetch user information based on ID
                  const user = await prisma.user.findUnique({
                      where: { id: req.user?.id }
                  });

                  if (user && user.name) {
                      uploadedBy = user.name;
                  } else {
                      console.warn('User not found or missing name with id:', req.user?.id);
                  }
              } catch (error) {
                  console.error('Error fetching user data:', error);
                  uploadedBy = 'User'; // Fallback in case of error
              }
          }

          // Save document metadata to the database
          const documentData: Document = {
              id: nanoid(),
              name: sanitizedFilename,
              type: req.body.type,
              fileUrl: fileUrl,
              fileType: fileExtension.slice(1),
              size: (req.file.size / 1024 / 1024).toFixed(2) + ' MB',
              uploadedOn: new Date(),
              uploadedBy: uploadedBy,
              leadId: leadId,
          };

        const newDocument = await this.documentService.createDocument(documentData);
        res.status(201).json(newDocument);
      } catch (uploadError) {
        console.error("Filezilla Upload Error:", uploadError);
        res.status(500).json({ message: "Failed to upload file to Filezilla." });
      }
    } catch (error) {
      console.error('Error uploading document:', error);
      next(error);
    }
  }

  public async getDocumentsForLead(req: Request, res: Response, next: NextFunction): Promise<void> {
    try {
      const leadId = req.params.leadId;
      const documents = await this.documentService.getDocumentsForLead(leadId);
      res.json(documents);
    } catch (error) {
      console.error('Error getting documents for lead:', error);
      next(error);
    }
  }

  public async deleteDocument(req: Request, res: Response, next: NextFunction): Promise<void> {
    try {
      const documentId = req.params.id;
      await this.documentService.deleteDocument(documentId);
      res.status(204).send(); // 204 No Content - success, no content to return
    } catch (error) {
      console.error('Error deleting document:', error);
      next(error);
    }
  }
}