import { NextFunction, Request, Response } from "express";
import { nanoid } from "nanoid";
import { Lead } from "../../interfaces/Lead";
import { UserLeadService } from "../../services/user/UserLeadService";
import { ParamsDictionary } from "express-serve-static-core";
import { ParsedQs } from "qs";
import { LeadStage } from "@prisma/client";

export class UserLeadController {
  private userLeadService: UserLeadService;

  constructor(userLeadService: UserLeadService) {
    this.userLeadService = userLeadService;
  }

  public async createLead(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const empCode = req.decoded?.userId as string; // Access from JWT

      console.log("Employee Code", empCode);

      if (!empCode) {
        res
          .status(401)
          .json({ message: "Unauthorized: Missing empCode in token" });
        return;
      }
      const travelDateFrom = req.body.travelDateFrom;
      const travelDateTo = req.body.travelDateTo;

      const leadData: Lead = {
        id: nanoid(),
        empCode: req.body.empCode, // Use the logged-in user's empCode
        agentName: req.body.agentName || null,
        agentCode: req.body.agentCode || null,
        leadNo: req.body.leadNo,
        travelDateFrom: travelDateFrom,
        travelDateTo: travelDateTo,
        totalPax: req.body.totalPax || null,
        description: req.body.description || null,
        customerName: req.body.customerName || null,
        email: req.body.email || null,
        phone: req.body.phone || null,
        adults: req.body.adults || null,
        kids: req.body.kids || null,
        infants: req.body.infants || null,
        remarks: req.body.remarks || null,
        agentId: req.body.agentId || null,
        userId:req.body.userId || null, // Use the logged-in user's ID
      };
      console.log("Lead user data", leadData);
      const newLead = await this.userLeadService.createLead(leadData);

      // ✅ Automatically set stage to 'new_lead' and create a follow-up
      // await this.userLeadService.changeStage(
      //   newLead.id,
      //   LeadStage.new_lead,  // ✅ Corrected property name
      //   req.decoded?.name as string, // Or use the user's name from the request
      //   "Lead Created" // Initial note
      // );

      res.status(201).json(newLead);
    } catch (error) {
      console.error("Error creating lead (user):", error);
      next(error);
    }
  }

  public async getUserLeads(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const userId = req.decoded?.userId as string;

      if (!userId) {
        res
          .status(401)
          .json({ message: "Unauthorized: Missing userId in token" });
        return;
      }

      const leads = await this.userLeadService.getUserLeads(userId);

      // console.log("User leads:", leads);
      res.json(leads);
    } catch (error) {
      console.error("Error getting user leads:", error);
      next(error);
    }
  }

  public async getLeadById(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const leadId = req.params.leadId;
    
      const lead = await this.userLeadService.getLeadById(leadId);

      if (!lead) {
        res.status(404).json([]);
        return;
      }

      res.json(lead);
    } catch (error) {
      console.error("Error getting lead by ID (user):", error);
      next(error);
    }
  }

  public async updateLead(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const leadId = req.params.leadId;
      // Access from JWT

      const travelDateFrom = req.body.travelDateFrom
        ? new Date(req.body.travelDateFrom).toISOString()
        : undefined;
      const travelDateTo = req.body.travelDateTo
        ? new Date(req.body.travelDateTo).toISOString()
        : undefined;

      const leadData: Partial<Lead> = {
        empCode: req.body.empCode || null,
        agentName: req.body.agentName,
        agentCode: req.body.agentCode,
        leadNo: req.body.leadNo,
        travelDateFrom: travelDateFrom,
        travelDateTo: travelDateTo,
        totalPax: req.body.totalPax,
        description: req.body.description,
        customerName: req.body.customerName,
        email: req.body.email,
        phone: req.body.phone,
        adults: Number(req.body.adults), // ✅ Convert to number
        kids: Number(req.body.kids),
        infants: Number(req.body.infants),
        remarks: req.body.remarks,
        agentId: req.body.agentId,
      };
      console.log("Update Lead user data", leadData);
      const updatedLead = await this.userLeadService.updateLead(
        leadId,
        leadData
      );

      if (updatedLead) {
        res.json(updatedLead);
      } else {
        res.status(404).json({ message: "Lead not found or unauthorized" });
      }
    } catch (error) {
      console.error("Error updating lead (user):", error);
      next(error);
    }
  }

  public async deleteLead(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const leadId = req.params.leadId;

      await this.userLeadService.deleteLead(leadId);
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting lead (user):", error);
      next(error);
    }
  }

  public async updateStatus(req: Request, res: Response, next: NextFunction) {
    const { id } = req.params;
    const { status } = req.body;
    try {
      const lead = await this.userLeadService.changeStatus(id, status);
      if (!lead) {
        return res.status(404).json({ message: "Lead not found" });
      }
      return res.status(200).json(lead);
    } catch (err) {
      console.error("Error updating lead status:", err);
      return res.status(500).json({ message: "Server error", error: err });
    }
  }

  public async updateStage(req: Request, res: Response, next: NextFunction) {
    const { id } = req.params;
    console.log("Request body in controller:", req.body);
    const { stage, notes, userName } = req.body;

    try {
      const lead = await this.userLeadService.changeStage(id, stage, userName, notes);
      if (!lead) {
        return res.status(404).json({ message: "Lead not found" });
      }
      return res.status(200).json(lead);
    } catch (err) {
      console.error("Error updating lead stage:", err);
      return res.status(500).json({ message: "Server error", error: err });
    }
  }
}