// backend/src/controllers/agentController.ts

import { NextFunction, Request, Response } from "express";
import { Agent, AgentCredentials } from "../../interfaces/Agent";
import { agentService } from "../../services/agent/agentService";

import fs from "fs";
import path from "path";
import {
  cloudConfig,
  uploadToCloud,
} from "../../config/Agent/cloudStorageConfigFile";
import { HttpError } from "../../utils/HttpError";

export class AgentController {
  private agentService: agentService;

  constructor(agentService: agentService) {
    this.agentService = agentService;
  }

  public async createAgent(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agentData: Agent = {
        agentInfo: null,
        agentCode: req.body.agentCode,
        companyName: req.body.companyName,
        contactPerson: req.body.contactPerson,
        designation: req.body.designation,
        companyAddress: req.body.companyAddress,
        state: req.body.state,
        email: req.body.email,
        phone: req.body.phone,
        markup: parseFloat(req.body.markup),
        agentPrivilege: req.body.agentPrivilege === "true",
      } as Agent;

      const credentialsData: AgentCredentials = {
        username: req.body.username,
        password: req.body.password,
      } as AgentCredentials;

      if (req.file) {
        const localFilePath = req.file.path;
        const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
        const remoteFileName = `agent-${uniqueSuffix}${path.extname(
          req.file.originalname
        )}`;
        await uploadToCloud(localFilePath, remoteFileName);
        credentialsData.filename = cloudConfig.baseUrl + remoteFileName;
        fs.unlink(
          localFilePath,
          (err) => err && console.error("Error deleting local file:", err)
        );
      }

      const newAgent = await this.agentService.createAgent(
        agentData,
        credentialsData
      );
      res.status(201).json(newAgent);
    } catch (error: any) {
      if (error instanceof HttpError) {
        res.status(error.status).json({ message: error.message });
      } else {
        console.error("Unexpected error creating agent:", error);
        res.status(500).json({ message: "Internal server error" });
      }
    }
  }

  public async updateAgent(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agentId = req.params.id;

      const agentData: Partial<Agent> = {
        agentCode: req.body.agentCode,
        companyName: req.body.companyName,
        contactPerson: req.body.contactPerson,
        designation: req.body.designation,
        companyAddress: req.body.companyAddress,
        state: req.body.state,
        email: req.body.email,
        phone: req.body.phone,
        markup: parseFloat(req.body.markup),
        agentPrivilege: req.body.agentPrivilege === "true",
        deleted: req.body.deleted === "true",
      };

      const credentialsData: Partial<AgentCredentials> = {
        username: req.body.username,
        password: req.body.password,
      };

      if (req.file) {
        const localFilePath = req.file.path;
        const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
        const remoteFileName = `agent-${uniqueSuffix}${path.extname(
          req.file.originalname
        )}`;
        await uploadToCloud(localFilePath, remoteFileName);
        credentialsData.filename = cloudConfig.baseUrl + remoteFileName;
        fs.unlink(
          localFilePath,
          (err) => err && console.error("Error deleting local file:", err)
        );
      }

      const updatedAgent = await this.agentService.updateAgent(
        agentId,
        agentData,
        credentialsData
      );

      if (!updatedAgent) {
        res.status(404).json({ message: "Agent not found" });
        return;
      }

      res.json(updatedAgent);
    } catch (error: any) {
      if (error instanceof HttpError) {
        res.status(error.status).json({ message: error.message });
      } else {
        console.error("Unexpected error updating agent:", error);
        res.status(500).json({ message: "Internal server error" });
      }
    }
  }

  public async getAllAgents(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agents = await this.agentService.getAllAgents();
      res.json(agents);
    } catch (error) {
      console.error("Error getting agents:", error);
      next(error);
    }
  }

  public async softDeleteAgent(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agentId = req.params.id;
      console.log("Controller - Soft Deleting Agent:", agentId);
      // Call the service to soft delete the agent

      const deletedAgent = await this.agentService.softDeleteAgent(agentId);
      if (!deletedAgent) {
        res.status(404).json({ message: "Agent not found" });
        return;
      }
      res.json(deletedAgent);
    } catch (error) {
      console.error("Error deleting agent:", error);
      next(error);
    }
  }

  public async restoreAgent(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agentId = req.params.id;
      const restoredAgent = await this.agentService.restoreAgent(agentId);
      if (!restoredAgent) {
        res.status(404).json({ message: "Agent not found for restore" });
        return;
      }
      res.json(restoredAgent);
    } catch (error) {
      console.error("Error restoring agent:", error);
      next(error);
    }
  }
  public async deleteAgent(
    req: Request,
    res: Response,
    next: NextFunction
  ): Promise<void> {
    try {
      const agentId = req.params.id;
      await this.agentService.deleteAgent(agentId);
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting agent:", error);
      next(error);
    }
  }
}
