import { Request, Response } from "express";
import { SupplierService } from "../../services/admin/supplierService";

const supplierService = new SupplierService();

export class SupplierController {
    async createSupplier(req: Request, res: Response) {
        try {
            const supplier = await supplierService.createSupplier(req.body);
            res.status(201).json(supplier);
        } catch (error: any) {
            console.error("Error creating supplier:", error);
            res.status(500).json({ message: error.message || "An unknown error occurred" });
        }
    }

    async getSuppliers(req: Request, res: Response) {
        try {
            const suppliers = await supplierService.getSuppliers();
            return res.status(200).json(suppliers || [])
        } catch (error: any) {
            console.error("Error getting suppliers:", error);
            res.status(500).json({ message: error.message || "An unknown error occurred" });
        }
    }

    async getSupplierById(req: Request, res: Response) {
        try {
            const id: string = req.params.id;  // Explicitly type `id`
            const supplier = await supplierService.getSupplierById(id);
            if (!supplier) {
                return res.status(404).json({ message: "Supplier not found" });
            }
            res.json(supplier);
        } catch (error: any) {
            console.error("Error getting supplier by ID:", error);
            res.status(500).json({ message: error.message || "An unknown error occurred" });
        }
    }

    async updateSupplier(req: Request, res: Response) {
        try {
            const id: string = req.params.id;  // Explicitly type `id`
            const supplier = await supplierService.updateSupplier(id, req.body);
            if (!supplier) {
                return res.status(404).json({ message: "Supplier not found" });
            }
            res.json(supplier);
        } catch (error: any) {
            console.error("Error updating supplier:", error);
            // Check for specific Prisma errors here if you want (e.g., record not found)
            res.status(500).json({ message: error.message || "An unknown error occurred" });
        }
    }

    async deleteSupplier(req: Request, res: Response) {
        try {
            const id: string = req.params.id;  // Explicitly type `id`
            await supplierService.deleteSupplier(id);
            res.json({ message: "Supplier deleted successfully" });
        } catch (error: any) {
            console.error("Error deleting supplier:", error);
            // Check for specific Prisma errors here if you want (e.g., record not found)
            res.status(500).json({ message: error.message || "An unknown error occurred" });
        }
    }
}
