// backend/src/controllers/packageController.ts

import { NextFunction, Request, Response } from "express";
import fs from 'fs';
import path from 'path';
import { Omit } from "utility-types";
import { cloudConfig, uploadToCloud } from "../../config/cloudStorageConfig";
import { Package } from "../../interfaces/Package";
import { PackageService } from "../../services/admin/packageService";

export class PackageController {
    private packageService: PackageService;

    constructor(packageService: PackageService) {
        this.packageService = packageService;
    }

    public async createPackage(
        req: Request,
        res: Response,
        next: NextFunction
    ): Promise<void> {
        try {
            console.log("Request Body:", req.body);
            const {
                packageName,
                tagLine,
                briefDescription,
                detailedDescription,
                terms,
                notes,
                remarks,
                starRating,
                price,
                hotel,
                daysNights,
                members,
                hotelId,
                imageUrl
            } = req.body

            const packageDataWithoutId: Omit<Package, "id"> = {
                packageName: packageName,
                tagLine: tagLine,
                briefDescription: briefDescription,
                detailedDescription: detailedDescription,
                terms: terms,
                notes: notes,
                remarks: remarks,
                imageUrl: imageUrl,
                starRating: parseInt(starRating),
                price: price,
                hotel: hotel,
                daysNights: daysNights,
                members: members,
                deleted: false,
                hotelId: hotelId,
            };

            const newPackage = await this.packageService.createPackage(
                packageDataWithoutId
            );
            res.status(201).json(newPackage);
            return;

        } catch (error) {
            console.error("Error creating package:", error);
            next(error);
            return;
        }
    }


    public async uploadImage(req: Request, res: Response, next: NextFunction): Promise<void> {
        try {
            if (!req.file) {
                res.status(400).json({ message: "No image file provided" });
                return;
            }

            const filename = req.file.filename;
            const localFilePath = req.file.path;
            const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
            const remoteFileName = `package-${uniqueSuffix}${path.extname(req.file.originalname)}`;

            try {
                await uploadToCloud(localFilePath, remoteFileName);
                const cloudImageUrl = cloudConfig.baseUrl + remoteFileName;

                fs.unlink(localFilePath, (err: NodeJS.ErrnoException | null) => {
                    if (err) {
                        console.error('Error deleting local file:', err);
                    } else {
                        console.log('Local file deleted successfully');
                    }
                });

                res.status(200).json({ imageUrl: cloudImageUrl });
            } catch (uploadError) {
                console.error("Cloud Upload Error:", uploadError);
                res.status(500).json({ message: "Failed to upload image to cloud." });
            }

        } catch (error) {
            console.error("Error uploading image:", error);
            next(error);
        }
    }


    public async updatePackage(
        req: Request,
        res: Response,
        next: NextFunction
    ): Promise<void> {
        try {
            const packageId = req.params.id;
            const filename = req.file ? req.file.filename : null;
            let cloudImageUrl: string | undefined;

            if (filename && req.file) {
                // Construct the local file path
                const localFilePath = req.file.path;

                const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
                const remoteFileName = `package-${uniqueSuffix}${path.extname(req.file.originalname)}`;
                // Upload to the cloud
                try {
                    await uploadToCloud(localFilePath, remoteFileName);
                    cloudImageUrl = cloudConfig.baseUrl + remoteFileName;

                    // Delete the local file after successful upload
                    fs.unlink(localFilePath, (err: NodeJS.ErrnoException | null) => {
                        if (err) {
                            console.error('Error deleting local file:', err);
                        } else {
                            console.log('Local file deleted successfully');
                        }
                    });

                } catch (uploadError) {
                    console.error("Cloud Upload Error:", uploadError);
                    // **IMPORTANT:** Send an error response *and* `return` to prevent further execution
                    res.status(500).json({ message: "Failed to upload image to cloud." });
                    return;  // <---- Add this return
                }
            }

            const packageData: Partial<Package> = {
                packageName: req.body.packageName,
                tagLine: req.body.tagLine,
                briefDescription: req.body.briefDescription,
                detailedDescription: req.body.detailedDescription,
                terms: req.body.terms,
                notes: req.body.notes,
                remarks: req.body.remarks,
                starRating: parseInt(req.body.starRating),
                price: req.body.price,
                hotel: req.body.hotel,
                daysNights: req.body.daysNights,
                members: req.body.members,
                hotelId: req.body.hotelId,
                // Only update imageUrl if a new file was uploaded.
                imageUrl: cloudImageUrl,
            };

            const updatedPackage = await this.packageService.updatePackage(
                packageId,
                packageData
            );
            res.json(updatedPackage);
            return;

        } catch (error) {
            console.error("Error updating package:", error);
            next(error);
            return;
        }
    }

    public async getPackages(
        req: Request,
        res: Response,
        next: NextFunction
    ): Promise<void> {
        try {
            const packages = await this.packageService.getAllPackages();
            res.json(packages);
            return;

        } catch (error) {
            console.error("Error getting packages:", error);
            next(error);
            return;
        }
    }

    public async getPackageById(
        req: Request,
        res: Response,
        next: NextFunction
    ): Promise<void> {
        try {
            const packageId = req.params.id;
            const packageItem = await this.packageService.getPackageById(packageId);
            if (!packageItem) {
                res.status(404).json({ message: "Package not found" });
                return;
            }
            console.log(packageItem);
            res.json(packageItem);
            return;

        } catch (error) {
            console.error("Error getting package by ID:", error);
            next(error);
            return;
        }
    }

    public async deletePackage(
        req: Request,
        res: Response,
        next: NextFunction
    ): Promise<void> {
        try {
            const packageId = req.params.id;
            await this.packageService.deletePackage(packageId);
            res.status(204).send(); // No content
            return;

        } catch (error) {
            console.error("Error deleting package:", error);
            next(error);
            return;
        }
    }
}
