import { NextFunction, Request, Response } from "express";
import { HotelService } from "../../services/admin/hotelService";

export class HotelController {
  private hotelService: HotelService;

  constructor(hotelService: HotelService) {
    this.hotelService = hotelService;
  }
  createHotel = async (req: Request, res: Response) => {
    try {
      const hotel = await this.hotelService.createHotel(req.body);
      res.status(201).json(hotel);
    } catch (error:any) {
      res.status(400).json({ error: error.message || "Failed to create hotel" });
    }
  };

  getHotels = async (req: Request, res: Response, next: unknown) => {
    try {
      const page = parseInt(req.query.page as string) || 1;
      const limit = parseInt(req.query.limit as string) || 10;
      const searchTerm = req.query.search as string || "";
  
      const hotels = await this.hotelService.getHotels(page, limit, searchTerm);
  
      res.json(hotels);
    } catch (error: any) {
      res.status(500).json({ error: error.message });
    }
  };
  
  

  getHotelById = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const hotel = await this.hotelService.getHotelById(req.params.id);
      if (!hotel) {
       res.status(404).json({ message: "Hotel not found" });
      }
      res.json(hotel);
    } catch (error) {
      next(error);
    }
  };

  updateHotel = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const hotel = await this.hotelService.updateHotel(req.params.id, req.body);
      console.log(hotel)
      res.json(hotel);
    } catch (error) {
      next(error);
    }
  };

  deleteHotel = async (req: Request, res: Response, next: NextFunction) => {
    try {
      await this.hotelService.deleteHotel(req.params.id);
      res.json({ message: "Hotel deleted successfully" });
    } catch (error) {
      next(error);
    }
  };

  createRoom=async (req: Request, res: Response)=> {
    try {
      const room = await this.hotelService.createRoom(req.body);
      console.log(room)
      res.status(201).json({ message: "Room created", room });
    } catch (error: any) {
       res.status(400).json({ error: error.message });
    }
  }

  getRoomById = async (req: Request, res: Response) => {
    try {
      const room = await this.hotelService.getRoomById(req.params.id);
      if (!room) {
        res.status(404).json({ message: "Room not found" });
      }
      res.json(room);
    } catch (error) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  };

  updateRoom = async (req: Request, res: Response) => {
    try {
      const room = await this.hotelService.updateRoom(req.params.id, req.body);
      res.json(room);
    } catch (error) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }

  deleteRoom = async (req: Request, res: Response) => { 
    try {
      await this.hotelService.deleteRoom(req.params.id);
      res.json({ message: "Room deleted successfully" });
    } catch (error) {
      if (error instanceof Error) {
        res.status(500).json({ message: error.message });
      } else {
        res.status(500).json({ message: "An unknown error occurred" });
      }
    }
  }


  getRoomsByHotel = async (req: Request, res: Response) => {
    try {
      const { hotelId } = req.params;
  
      if (!hotelId) {
        res.status(400).json({ message: 'Hotel ID is required' });
      }
  
      const rooms = await this.hotelService.getRoomsByHotelId(hotelId);
      res.status(200).json(rooms);
    } catch (error: any) {
      res.status(500).json({ message: error.message });
    }
}
}
