// backend/src/controllers/admin/bookingController.ts
import { Request, Response } from "express";
import { BookingService } from "../../services/admin/bookingService";
// Import Prisma types if you need to cast req.body specifically for priceDetailsAccommodation
import { Prisma } from '@prisma/client';

export class BookingController {
  private bookingService: BookingService;

  constructor(bookingService?: BookingService) {
    this.bookingService = bookingService || new BookingService();
  }

  async fetchQuotationWithDetails(req: Request, res: Response) {
    try {
      const { id } = req.params;
      if (!id) {
        return res.status(400).json({ error: "Quotation ID is required" });
      }
      const quotationDetails = await this.bookingService.fetchQuotationWithDetails(id);
      return res.status(200).json(quotationDetails);
    } catch (error: any) {
      console.error("Error fetching quotation details:", error.message);
      return res.status(500).json({ error: error.message || "Failed to fetch quotation details" });
    }
  }

  async createBooking(req: Request, res: Response) {
    try {
      const { quotationId } = req.body;
      if (!quotationId) {
        return res.status(400).json({ error: "quotationId is required" });
      }
      const booking = await this.bookingService.createBookingFromQuotation(quotationId);
      return res.status(201).json(booking);
    } catch (error: any) {
      console.error("Error creating booking:", error.message);
      return res.status(500).json({ error: error.message || "Failed to create booking" });
    }
  }

  async getBookings(req: Request, res: Response) {
    try {
      const bookings = await this.bookingService.getAllBookings();
      return res.status(200).json(bookings || []);
    } catch (error: any) {
      console.error("Error fetching bookings:", error.message); // Changed from quotations
      res.status(500).json({ error: "Internal Server Error" });
    }
  }

  async getBookingById(req: Request, res: Response) {
    try {
      const { id } = req.params;
      // console.log("Fetching booking for ID:", id); // Changed from quotation

      if (!id) {
        return res.status(400).json({ error: "Booking ID is required" });
      }
      const booking = await this.bookingService.getBookingById(id);
      // console.log(booking, "Booking in controller");
      // No need to check !booking here if service throws error for "Booking not found"
      res.status(200).json(booking);
    } catch (error: any) {
      console.error("Controller error getBookingById:", error.message);
      if (error.message === "Booking not found") {
        return res.status(404).json({ error: "Booking not found" });
      }
      res.status(500).json({ error: "Internal Server Error" });
    }
  }

  async updateBookingStatus(req: Request, res: Response) {
    try {
      const { id } = req.params;
      const { status, userName } = req.body; // Ensure userName is sent if needed by service/repo
      if (!id) return res.status(400).json({ error: "Booking ID is required" });
      if (!status) return res.status(400).json({ error: "New status is required" });

      const updatedBooking = await this.bookingService.updateBookingStatus(id, status, userName);
      res.status(200).json(updatedBooking);
    } catch (error: any) {
      console.error("Error updating booking status:", error.message);
      res.status(500).json({ error: error.message || "Failed to update booking status" });
    }
  }

  async updateBookingStage(req: Request, res: Response) {
    try {
      const { id } = req.params;
      const { stage, notes, userName } = req.body; // Ensure these are sent
      if (!id) return res.status(400).json({ error: "Booking ID is required" });
      if (!stage) return res.status(400).json({ error: "New stage is required" });

      const updatedBooking = await this.bookingService.updateBookingStage(id, stage, notes, userName);
      res.status(200).json(updatedBooking);
    } catch (error: any) {
      console.error("Error updating booking stage:", error.message);
      res.status(500).json({ error: error.message || "Failed to update booking stage" });
    }
  }

  async updateAccommodationHandler(req: Request, res: Response) {
    const { bookingId, accommodationId } = req.params;
    // Destructure all expected fields from req.body
    const {
      hotel,
      room,
      supplierName,
      supplierContact,
      travelDate,
      nights,
      quotePrice,
      actualPrice,
      confirmationNo, // This is what the service expects
      details,
      quantity,
      priceDetailsAccommodation // This should be an array of objects
    } = req.body;

    // console.log("Updating accommodation with data in controller:", req.body);

    try {
      // Ensure priceDetailsAccommodation is correctly typed if it needs specific fields from Prisma.
      // For createMany, it's usually an array of objects matching the model fields (excluding id).
      const updatePayload = {
        hotel,
        room,
        supplierName,
        supplierContact,
        travelDate,
        nights,
        quotePrice,
        actualPrice,
        confirmationNo, // Pass this to the service
        details,
        quantity,
        priceDetailsAccommodation: priceDetailsAccommodation as Prisma.BookingPriceDetailsAccommodationCreateManyAccommodationInput[] | undefined
      };

      const result = await this.bookingService.updateAccommodationInBooking(
        bookingId,
        accommodationId,
        updatePayload
      );
      res.status(200).json({ message: "Accommodation updated successfully", data: result });
    } catch (error: any) {
      console.error("Controller error updateAccommodationHandler:", error.message);
      res.status(500).json({ error: error.message || "Internal server error" });
    }
  }

  async updateItinerary(req: Request, res: Response) {
    try {
      const { bookingId, itineraryId } = req.params;
      // console.log("Updating itinerary with bookingId:", bookingId, "and itineraryId:", itineraryId);
      const updateData = req.body; // Assuming req.body matches ItineraryUpdateServiceData DTO

      const updatedItinerary = await this.bookingService.updateItinerary(
        bookingId,
        itineraryId,
        updateData
      );

      res.json({
        success: true,
        message: "Itinerary updated successfully",
        data: updatedItinerary
      });
    } catch (error: any) {
      console.error("Controller error updateItinerary:", error.message);
      res.status(500).json({ success: false, error: error.message || "Failed to update itinerary" });
    }
  }

  async getFinalItineraryItems(req: Request, res: Response) {
    try {
      const { bookingId } = req.params;
      const items = await this.bookingService.getFinalItineraryItems(bookingId);
      res.status(200).json(items);
    } catch (error: any) {
      console.error("Error fetching final itinerary items:", error.message);
      res.status(500).json({ error: error.message || "Failed to fetch final itinerary items" });
    }
  }

  async createFinalItineraryItem(req: Request, res: Response) {
    try {
      const { bookingId } = req.params;
      const data = req.body; // This should match Omit<FinalItineraryItem, ...>
      const newItem = await this.bookingService.createFinalItineraryItem(bookingId, data);
      res.status(201).json(newItem);
    } catch (error: any) {
      console.error("Error creating final itinerary item:", error.message);
      res.status(500).json({ error: error.message || "Failed to create final itinerary item" });
    }
  }

  async updateFinalItineraryItem(req: Request, res: Response) {
    try {
      const { itemId } = req.params;
      const data = req.body; // This should match Partial<Omit<FinalItineraryItem, ...>>
      const updatedItem = await this.bookingService.updateFinalItineraryItem(itemId, data);
      res.status(200).json(updatedItem);
    } catch (error: any) {
      console.error("Error updating final itinerary item:", error.message);
      res.status(500).json({ error: error.message || "Failed to update final itinerary item" });
    }
  }

  async deleteFinalItineraryItem(req: Request, res: Response) {
    try {
      const { itemId } = req.params;
      await this.bookingService.deleteFinalItineraryItem(itemId);
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting final itinerary item:", error.message);
      res.status(500).json({ error: error.message || "Failed to delete final itinerary item" });
    }
  }

  async generateInitialFinalItinerary(req: Request, res: Response) {
    try {
      const { bookingId } = req.params;
      const items = await this.bookingService.generateInitialFinalItinerary(bookingId);
      res.status(200).json({ message: "Initial final itinerary generated successfully", data: items });
    } catch (error: any) {
      console.error("Error generating initial final itinerary:", error.message);
      res.status(500).json({ error: error.message || "Failed to generate initial final itinerary" });
    }
  }
}