import { isValid } from 'date-fns';
import { NextFunction, Request, Response } from "express";
import prisma from "../../config/prismaClient";
import { Attraction } from "../../interfaces/Attraction";
import { AttractionService } from "../../services/admin/attractionService";

export class AttractionController {
    private attractionService: AttractionService;

    constructor() {
        this.attractionService = new AttractionService();
    }

    createAttraction = async (req: Request, res: Response, next: NextFunction) => {
        try {
            const attraction: Attraction = req.body;

            // *** DATA VALIDATION HERE!! ****  IMPORTANT!!!
            if (!attraction.name || attraction.name.length < 3) {
                return res.status(400).json({ message: "Name must be at least 3 characters long." });
            }
            if (!attraction.supplierId) {
                return res.status(400).json({ message: "Supplier ID is required." });
            }

            // *** VALIDATE SUPPLIER ID ***
            const supplierExists = await prisma.suppliers.findUnique({
                where: {
                    id: attraction.supplierId,
                },
            });

            if (!supplierExists) {
                return res.status(400).json({ message: "Invalid supplier ID." });
            }

          // Date validation
            if (attraction.surchargeDates) {
                for (const dateString of attraction.surchargeDates) {
                    if (!isValid(new Date(dateString))) {
                        return res.status(400).json({ message: `Invalid surcharge date: ${dateString}` });
                    }
                }
            }

            if (attraction.closedDates) {
                for (const dateString of attraction.closedDates) {
                    if (!isValid(new Date(dateString))) {
                        return res.status(400).json({ message: `Invalid closed date: ${dateString}` });
                    }
                }
            }
            const createdAttraction = await this.attractionService.createAttraction(attraction);
            res.status(201).json(createdAttraction);
        } catch (error: any) {
            console.error("Error creating attraction in controller:", error);
            res.status(500).json({ message: error.message || "Internal Server Error" });
        }
    };

    getAttractions = async (req: Request, res: Response, next: NextFunction) => {
        try {
            const attractions = await this.attractionService.getAttractions();
            res.json(attractions);
        } catch (error) {
            next(error);
        }
    };

    getAttractionById = async (req: Request, res: Response, next: NextFunction) => {
        try {
            const attraction = await this.attractionService.getAttractionById(req.params.id);
            if (!attraction) {
                res.status(404).json({ message: "Attraction not found" });
                return;
            }
            res.json(attraction);
        } catch (error) {
            next(error);
        }
    };

    updateAttraction = async (req: Request, res: Response, next: NextFunction) => {
        try {
            const attraction = await this.attractionService.updateAttraction(req.params.id, req.body);
            res.json(attraction);
        } catch (error) {
            next(error);
        }
    };

    deleteAttraction = async (req: Request, res: Response, next: NextFunction) => {
        try {
            await this.attractionService.deleteAttraction(req.params.id);
            res.json({ message: "Attraction deleted successfully" });
        } catch (error) {
            next(error);
        }
    };
}
