"use strict";
// backend/src/services/agentService.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.agentService = void 0;
const client_1 = require("@prisma/client"); // Import Prisma types
const bcryptjs_1 = __importDefault(require("bcryptjs"));
// import { AgentRepository } from "../../repositories/agent/agentRepository";
// import { AgentRepository } from "../../repositories/user/userAgentRepository";
class agentService {
    constructor(agentRepository) {
        this.agentRepository = agentRepository;
    }
    async createAgent(agentData, credentialsData = {}) {
        try {
            console.log("Service - Creating Agent:", agentData);
            if (!agentData.email) {
                throw new Error("Agent email is required.");
            }
            // Validate password if provided
            if (credentialsData.password && credentialsData.password.length < 6) {
                throw new Error("Password must be at least 6 characters long.");
            }
            // Hash password if it exists
            if (credentialsData.password) {
                credentialsData.password = await bcryptjs_1.default.hash(credentialsData.password, 10);
            }
            else if (agentData.agentPrivilege) {
                console.warn(`Agent ${agentData.email} created with privileges but no password.`);
            }
            // Call repository to save the agent
            const newAgent = await this.agentRepository.createAgent(agentData, credentialsData);
            return newAgent;
        }
        catch (error) {
            console.error("Error creating agent in service:", error);
            throw error;
        }
    }
    async updateAgent(agentId, agentData, credentialsData) {
        var _a;
        try {
            // Hash password if provided in the update
            if (credentialsData === null || credentialsData === void 0 ? void 0 : credentialsData.password) {
                if (credentialsData.password.length < 6) {
                    throw new Error("Password must be at least 6 characters long.");
                }
                const hashedPassword = await bcryptjs_1.default.hash(credentialsData.password, 10); // Hash with salt rounds
                credentialsData.password = hashedPassword; // Replace plain password with hash
            }
            const updatedAgent = await this.agentRepository.updateAgent(agentId, agentData, credentialsData);
            if (!updatedAgent) {
                throw new Error(`Agent with ID ${agentId} not found for update.`);
            }
            return updatedAgent;
        }
        catch (error) {
            console.error("Error updating agent in service:", error);
            // Handle potential unique constraint errors during update
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2002") {
                const fields = (_a = error.meta) === null || _a === void 0 ? void 0 : _a.target;
                throw new Error(`Agent update failed: The following field(s) must be unique: ${fields === null || fields === void 0 ? void 0 : fields.join(", ")}`);
            }
            // Check if it's a Prisma specific error for not found (already handled by repo returning null/throwing)
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025") {
                throw new Error(`Agent update failed: Record to update not found.`);
            }
            throw error; // Re-throw other errors
        }
    }
    async getAllAgents() {
        try {
            const agents = await this.agentRepository.getAllAgents();
            return agents;
        }
        catch (error) {
            console.error("Error getting agents in service:", error);
            throw error;
        }
    }
    async softDeleteAgent(agentId) {
        console.log("Service - Soft Deleting Agent:", agentId);
        try {
            const deletedAgent = await this.agentRepository.softDeleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        }
        catch (error) {
            console.error("Error soft deleting agent in service:", error);
            // Handle potential unique constraint errors during deletion
            return null; // Ensure a return value in case of an error
        }
    }
    async restoreAgent(agentId) {
        try {
            const restoredAgent = await this.agentRepository.restoreAgent(agentId);
            if (!restoredAgent) {
                console.warn(`Agent with ID ${agentId} not found for restore.`);
                return null;
            }
            return restoredAgent;
        }
        catch (error) {
            console.error("Error restoring agent in service:", error);
            throw error;
        }
    }
    async deleteAgent(agentId) {
        try {
            const deletedAgent = await this.agentRepository.deleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        }
        catch (error) {
            console.error("Error deleting agent in service:", error);
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025") {
                console.warn(`   Agent with ID ${agentId} not found for deletion (Prisma Error).`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            throw error;
        }
    }
}
exports.agentService = agentService;
//# sourceMappingURL=userAgentService.js.map