"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BookingService = void 0;
//userBookingService
const UserbookingRepository_1 = require("../../repositories/user/UserbookingRepository");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const client_1 = require("@prisma/client"); // Import LeadStatus enum
class BookingService {
    constructor(bookingRepository) {
        this.bookingRepository = bookingRepository || new UserbookingRepository_1.BookingRepository();
    }
    async fetchQuotationWithDetails(quotationId) {
        try {
            return await this.bookingRepository.fetchQuotationWithDetails(quotationId);
        }
        catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }
    async createBookingFromQuotation(quotationId) {
        try {
            const quotation = await this.bookingRepository.fetchQuotationWithDetails(quotationId);
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            return await this.bookingRepository.handleCreateBooking(quotation);
        }
        catch (error) {
            console.error("Error creating booking from quotation:", error);
            throw new Error("Failed to create booking from quotation");
        }
    }
    async getAllBookings(userId) {
        try {
            return await this.bookingRepository.getAllBookings(userId);
        }
        catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve bookings");
        }
    }
    async getBookingById(id) {
        try {
            return await this.bookingRepository.getBookingById(id);
        }
        catch (error) {
            console.error("Error fetching booking by ID:", error);
            throw new Error("Failed to retrieve booking");
        }
    }
    async updateBookingStatus(bookingId, status, userName) {
        try {
            if (!Object.values(client_1.LeadStatus).includes(status)) {
                throw new Error('Invalid status value');
            }
            const updatedBooking = await prismaClient_1.default.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            status: status,
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                },
            });
            return updatedBooking;
        }
        catch (error) {
            console.error('Error updating booking status:', error);
            throw new Error('Failed to update booking status');
        }
    }
    async updateBookingStage(bookingId, stage, notes, userName) {
        try {
            if (!Object.values(client_1.LeadStage).includes(stage)) {
                throw new Error('Invalid stage value');
            }
            const updatedBooking = await prismaClient_1.default.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            stage: stage,
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                },
            });
            return updatedBooking;
        }
        catch (error) {
            console.error('Error updating booking stage:', error);
            throw new Error('Failed to update booking stage');
        }
    }
    async updateAccommodationInBooking(bookingId, accommodationId, updateData) {
        try {
            const booking = await this.bookingRepository.findBookingWithAccommodations(bookingId);
            if (!booking)
                throw new Error("Booking not found");
            const accommodation = booking.accommodations.find((acc) => acc.id === accommodationId);
            if (!accommodation)
                throw new Error("Accommodation not found in this booking");
            const updated = await this.bookingRepository.updateAccommodation(accommodationId, updateData);
            return updated;
        }
        catch (error) {
            console.error("Service Error:", error);
            throw new Error("Failed to update accommodation in booking");
        }
    }
    async updateItinerary(bookingId, itineraryId, updateData) {
        // Verify booking exists
        const booking = await this.bookingRepository.getBookingById(bookingId);
        if (!booking) {
            throw new Error("Booking not found");
        }
        // Verify itinerary belongs to this booking
        const itinerary = booking.itineraries.find(it => it.id === itineraryId);
        console.log("Itinerary found:", itinerary);
        if (!itinerary) {
            throw new Error("Itinerary not found in this booking");
        }
        return this.bookingRepository.updateItinerary(itineraryId, updateData);
    }
}
exports.BookingService = BookingService;
//# sourceMappingURL=UserBookingService.js.map