"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.agentService = void 0;
const client_1 = require("@prisma/client");
const HttpError_1 = require("../../utils/HttpError");
const bcrypt_1 = __importDefault(require("bcrypt")); // Import bcrypt
class agentService {
    constructor(agentRepository) {
        this.agentRepository = agentRepository;
    }
    async createAgent(agentData, credentialsData) {
        try {
            // 🔍 Check email conflict
            if (agentData.email) {
                const existingEmail = await this.agentRepository.getAgentByEmail(agentData.email);
                if (existingEmail) {
                    throw new HttpError_1.HttpError("An agent with this email already exists.", 409);
                }
            }
            // 🔍 Check phone conflict
            if (agentData.phone) {
                const existingPhone = await this.agentRepository.getAgentByPhone(agentData.phone);
                if (existingPhone) {
                    throw new HttpError_1.HttpError("An agent with this phone number already exists.", 409);
                }
            }
            // 🔍 Check agentCode conflict
            if (agentData.agentCode) {
                const existingCode = await this.agentRepository.getAgentByAgentCode(agentData.agentCode);
                if (existingCode) {
                    throw new HttpError_1.HttpError("An agent with this agent code already exists.", 409);
                }
            }
            // Hash the password if credentials are provided
            if (credentialsData === null || credentialsData === void 0 ? void 0 : credentialsData.password) {
                const hashedPassword = await bcrypt_1.default.hash(credentialsData.password, 10); // 10 is the salt rounds
                credentialsData.password = hashedPassword; // Store the HASHED password
            }
            return await this.agentRepository.createAgent(agentData, credentialsData);
        }
        catch (error) {
            if (error instanceof HttpError_1.HttpError) {
                throw error; // Re-throw HttpError
            }
            else {
                console.error("AgentService Create Error:", error);
                throw new HttpError_1.HttpError("Failed to create agent", 500);
            }
        }
    }
    async updateAgent(agentId, agentData, credentialsData) {
        try {
            // Conditionally hash the password only if a new password is provided
            if (credentialsData === null || credentialsData === void 0 ? void 0 : credentialsData.password) {
                const hashedPassword = await bcrypt_1.default.hash(credentialsData.password, 10);
                credentialsData.password = hashedPassword; // Store the HASHED password
            }
            else {
                credentialsData === null || credentialsData === void 0 ? true : delete credentialsData.password;
            }
            return await this.agentRepository.updateAgent(agentId, agentData, credentialsData);
        }
        catch (error) {
            if (error instanceof HttpError_1.HttpError) {
                throw error; // Re-throw HttpError
            }
            else {
                console.error("AgentService Update Error:", error);
                throw new HttpError_1.HttpError("Failed to update agent", 500);
            }
        }
    }
    async getAllAgents() {
        try {
            const agents = await this.agentRepository.getAllAgents();
            return agents;
        }
        catch (error) {
            console.error("Error getting agents in service:", error);
            throw error;
        }
    }
    async softDeleteAgent(agentId) {
        console.log("Service - Soft Deleting Agent:", agentId);
        try {
            const deletedAgent = await this.agentRepository.softDeleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        }
        catch (error) {
            console.error("Error soft deleting agent in service:", error);
            // Handle potential unique constraint errors during deletion
            return null; // Ensure a return value in case of an error
        }
    }
    async restoreAgent(agentId) {
        try {
            const restoredAgent = await this.agentRepository.restoreAgent(agentId);
            if (!restoredAgent) {
                console.warn(`Agent with ID ${agentId} not found for restore.`);
                return null;
            }
            return restoredAgent;
        }
        catch (error) {
            console.error("Error restoring agent in service:", error);
            throw error;
        }
    }
    async deleteAgent(agentId) {
        try {
            const deletedAgent = await this.agentRepository.deleteAgent(agentId);
            if (!deletedAgent) {
                // Repository might return null if not found, handle it here
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            return deletedAgent;
        }
        catch (error) {
            console.error("Error deleting agent in service:", error);
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025") {
                console.warn(`   Agent with ID ${agentId} not found for deletion (Prisma Error).`);
                return null;
                // Or throw: throw new Error(Agent with ID ${agentId} not found for deletion.);
            }
            throw error;
        }
    }
}
exports.agentService = agentService;
//# sourceMappingURL=agentService.js.map