"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BookingService = void 0;
// backend/src/services/admin/bookingService.ts
const bookingRepository_1 = require("../../repositories/admin/bookingRepository");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const client_1 = require("@prisma/client");
class BookingService {
    constructor(bookingRepository) {
        this.bookingRepository = bookingRepository || new bookingRepository_1.BookingRepository();
    }
    async fetchQuotationWithDetails(quotationId) {
        try {
            return await this.bookingRepository.fetchQuotationWithDetails(quotationId);
        }
        catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }
    async createBookingFromQuotation(quotationId) {
        try {
            const quotation = await this.bookingRepository.fetchQuotationWithDetails(quotationId);
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            return await this.bookingRepository.handleCreateBooking(quotation);
        }
        catch (error) {
            console.error("Error creating booking from quotation:", error);
            throw new Error("Failed to create booking from quotation");
        }
    }
    async getAllBookings() {
        try {
            return await this.bookingRepository.getAllBookings();
        }
        catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve bookings");
        }
    }
    async getBookingById(id) {
        try {
            const booking = await this.bookingRepository.getBookingById(id);
            if (!booking) {
                throw new Error("Booking not found");
            }
            return booking;
        }
        catch (error) { // Catch specific error type if possible
            console.error("Error fetching booking by ID in service:", error.message);
            if (error.message === "Booking not found") {
                throw error;
            }
            throw new Error("Failed to retrieve booking");
        }
    }
    async updateBookingStatus(bookingId, status, userName) {
        try {
            if (!Object.values(client_1.LeadStatus).includes(status)) {
                throw new Error('Invalid status value');
            }
            // Ensure your prisma.booking.update includes the necessary relations if you return them
            return await prismaClient_1.default.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            status: status,
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    hotelVouchers: true,
                },
            });
        }
        catch (error) {
            console.error('Error updating booking status:', error);
            throw new Error('Failed to update booking status');
        }
    }
    async updateBookingStage(bookingId, stage, notes, userName) {
        try {
            if (!Object.values(client_1.LeadStage).includes(stage)) {
                throw new Error('Invalid stage value');
            }
            return await prismaClient_1.default.booking.update({
                where: { id: bookingId },
                data: {
                    lead: {
                        update: {
                            stage: stage,
                            // If you have a FollowUp model for stage changes, create it here
                        },
                    },
                },
                include: {
                    lead: true,
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    hotelVouchers: true,
                },
            });
        }
        catch (error) {
            console.error('Error updating booking stage:', error);
            throw new Error('Failed to update booking stage');
        }
    }
    // This is the critical method for the TS2554 error
    async updateAccommodationInBooking(bookingId, accommodationId, updateData // Using the defined type
    ) {
        try {
            const booking = await this.bookingRepository.findBookingWithAccommodations(bookingId);
            if (!booking)
                throw new Error("Booking not found");
            const accommodationExists = booking.accommodations.find((acc) => acc.id === accommodationId);
            if (!accommodationExists)
                throw new Error("Accommodation not found in this booking");
            // Pass data to repository, ensuring field names match (e.g., confirmationNo)
            // The repository's updateAccommodation expects an object that matches its BookingAccommodationUpdateData type
            return await this.bookingRepository.updateAccommodation(accommodationId, {
                hotel: updateData.hotel,
                room: updateData.room,
                supplierName: updateData.supplierName,
                supplierContact: updateData.supplierContact,
                travelDate: updateData.travelDate,
                nights: updateData.nights,
                quotePrice: updateData.quotePrice,
                actualPrice: updateData.actualPrice,
                confirmationNo: updateData.confirmationNo, // Map from service DTO to repo DTO if names differ
                details: updateData.details,
                quantity: updateData.quantity,
                priceDetailsAccommodation: updateData.priceDetailsAccommodation,
            });
        }
        catch (error) {
            console.error("Service Error in updateAccommodationInBooking:", error.message);
            throw new Error("Failed to update accommodation in booking");
        }
    }
    async updateItinerary(bookingId, itineraryId, updateData) {
        const booking = await this.bookingRepository.getBookingById(bookingId);
        if (!booking) {
            throw new Error("Booking not found");
        }
        const itineraryExists = booking.itineraries.find(it => it.id === itineraryId);
        if (!itineraryExists) {
            throw new Error("Itinerary not found in this booking");
        }
        return this.bookingRepository.updateItinerary(itineraryId, {
            date: updateData.date,
            label: updateData.label,
            category: updateData.category,
            supplierId: updateData.supplierId,
            supplierName: updateData.supplierName,
            supplierContact: updateData.supplierContact,
            quotePrice: updateData.quotePrice,
            actualPrice: updateData.actualPrice,
            confirmationNo: updateData.confirmationNo, // Repo expects confirmationNumber based on its DTO
            details: updateData.details,
            priceDetailsList: updateData.priceDetailsList,
        });
    }
    async getFinalItineraryItems(bookingId) {
        return this.bookingRepository.getFinalItineraryItemsByBookingId(bookingId);
    }
    async createFinalItineraryItem(bookingId, data) {
        return this.bookingRepository.createFinalItineraryItem(bookingId, data);
    }
    async updateFinalItineraryItem(itemId, data) {
        return this.bookingRepository.updateFinalItineraryItem(itemId, data);
    }
    async deleteFinalItineraryItem(itemId) {
        return this.bookingRepository.deleteFinalItineraryItem(itemId);
    }
    async generateInitialFinalItinerary(bookingId) {
        return this.bookingRepository.generateInitialFinalItineraryForBooking(bookingId);
    }
}
exports.BookingService = BookingService;
//# sourceMappingURL=bookingService.js.map