"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuotationRepository = void 0;
const client_1 = require("@prisma/client");
const notificationService_1 = require("../../services/common/notificationService");
const prisma = new client_1.PrismaClient();
class QuotationRepository {
    constructor() {
        this.notificationService = new notificationService_1.NotificationService(); // Initialize once
    }
    // Existing getDuplicateRefNo method (no changes needed here)
    getDuplicateRefNo(originalRefNo, existingRefNos) {
        // Remove any trailing -D{number} from the original refNo
        const baseRefNo = originalRefNo.replace(/-D\d+$/, "");
        // Find all duplicates of this base refNo
        const duplicates = existingRefNos.filter((ref) => ref.startsWith(`${baseRefNo}-D`));
        // Find highest -D{number}
        const maxDuplicateNumber = duplicates.reduce((max, ref) => {
            const match = ref.match(/-D(\d+)$/);
            const num = match ? parseInt(match[1], 10) : 0;
            return Math.max(max, num);
        }, 0);
        // Return next duplicate refNo
        return `${baseRefNo}-D${maxDuplicateNumber + 1}`;
    }
    async createQuotation(data, userid) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        try {
            const parseDate = (d) => d ? new Date(d) : null;
            const generateRefNo = async () => {
                var _a, _b;
                const leadId = (_b = (_a = data.leadRefNo) === null || _a === void 0 ? void 0 : _a.replace(/^LD-/, "")) !== null && _b !== void 0 ? _b : "UNKNOWN";
                const baseRef = `QT-${leadId}`;
                // Get only original and revised quotations (exclude duplicates like -D1, -D2)
                // This logic for `generateRefNo` is for revisions/new, not for -D duplicates.
                // It correctly finds the revision count based on QT-LEADID or QT-LEADID-R#
                const existingQuotations = await prisma.quotation.findMany({
                    where: {
                        leadRefNo: data.leadRefNo,
                        userid: userid, // Added userid filter
                        NOT: {
                            refNo: {
                                endsWith: '-D1', // This condition is too specific, needs to be more general
                            },
                        },
                    },
                    select: {
                        refNo: true,
                    },
                });
                const revisionCount = existingQuotations.filter(q => q.refNo === baseRef || (q.refNo.startsWith(`${baseRef}-R`) && !q.refNo.includes('-D'))).length;
                return revisionCount === 0 ? baseRef : `${baseRef}-R${revisionCount + 1}`;
            };
            let refNo;
            if ((_a = data.refNo) === null || _a === void 0 ? void 0 : _a.includes("-D")) {
                refNo = data.refNo; // This path won't be used by the duplicate function we are creating
            }
            else if (data.refNo) {
                // This implies 'data.refNo' is provided but doesn't have '-D', so it's a revision request
                refNo = await generateRefNo(); // Revision request
                console.log("Generated revision refNo:", refNo);
            }
            else {
                refNo = await generateRefNo(); // Fresh quotation (no refNo provided)
                console.log("Generated new refNo:", refNo);
            }
            // const newStage = "quotation_created"; //No lead update at user end
            const quotation = await prisma.quotation.create({
                data: {
                    refNo,
                    leadRefNo: data.leadRefNo,
                    leadAgent: (_b = data.leadAgent) !== null && _b !== void 0 ? _b : null,
                    status: (_c = data.status) !== null && _c !== void 0 ? _c : null,
                    guestName: (_d = data.guestName) !== null && _d !== void 0 ? _d : null,
                    remarks: (_e = data.remarks) !== null && _e !== void 0 ? _e : null,
                    currency: (_f = data.currency) !== null && _f !== void 0 ? _f : null,
                    totalBeforeMarkup: (_g = data.totalBeforeMarkup) !== null && _g !== void 0 ? _g : null,
                    totalAfterMarkup: (_h = data.totalAfterMarkup) !== null && _h !== void 0 ? _h : null,
                    travelDates: data.travelDates,
                    userid: userid, // Associate with user
                    agentId: (_j = data.agentId) !== null && _j !== void 0 ? _j : null,
                    accommodations: {
                        create: ((_k = data.accommodations) !== null && _k !== void 0 ? _k : []).map((acc) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
                            return ({
                                hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                                hotelAddress: (_b = acc.hotelAddress) !== null && _b !== void 0 ? _b : null,
                                supplierId: (_c = acc.supplierId) !== null && _c !== void 0 ? _c : null,
                                supplierName: (_d = acc.supplierName) !== null && _d !== void 0 ? _d : null,
                                supplierContact: (_e = acc.supplierContact) !== null && _e !== void 0 ? _e : null,
                                room: (_f = acc.room) !== null && _f !== void 0 ? _f : null,
                                travelDate: parseDate(acc.travelDate),
                                endDate: parseDate(acc.endDate),
                                nights: (_g = acc.nights) !== null && _g !== void 0 ? _g : null,
                                surchargeNights: (_h = acc.surchargeNights) !== null && _h !== void 0 ? _h : null,
                                quantity: (_j = acc.quantity) !== null && _j !== void 0 ? _j : null,
                                totalPrice: (_k = acc.totalPrice) !== null && _k !== void 0 ? _k : null,
                                totalPriceAfterMarkup: (_l = acc.totalPriceAfterMarkup) !== null && _l !== void 0 ? _l : null,
                                priceDetailsAccommodation: {
                                    create: ((_m = acc.priceDetails) !== null && _m !== void 0 ? _m : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                                        return ({
                                            roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                            category: (_b = pd.category) !== null && _b !== void 0 ? _b : null,
                                            pax: (_c = pd.pax) !== null && _c !== void 0 ? _c : null,
                                            price: (_d = pd.price) !== null && _d !== void 0 ? _d : null,
                                            priceAfterMarkup: (_e = pd.priceAfterMarkup) !== null && _e !== void 0 ? _e : null,
                                            costPrice: (_f = pd.costPrice) !== null && _f !== void 0 ? _f : null,
                                            optionalPrice: (_g = pd.optionalPrice) !== null && _g !== void 0 ? _g : null,
                                            quantity: (_h = pd.quantity) !== null && _h !== void 0 ? _h : null,
                                            totalPrice: (_j = pd.totalPrice) !== null && _j !== void 0 ? _j : null,
                                            totalPriceAfterMarkup: (_k = pd.totalPriceAfterMarkup) !== null && _k !== void 0 ? _k : null,
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                    itineraries: {
                        create: ((_l = data.itineraries) !== null && _l !== void 0 ? _l : []).map((itin) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h;
                            return ({
                                date: parseDate(itin.date),
                                label: (_a = itin.label) !== null && _a !== void 0 ? _a : null,
                                category: (_b = itin.category) !== null && _b !== void 0 ? _b : null,
                                supplierId: (_c = itin.supplierId) !== null && _c !== void 0 ? _c : null,
                                supplierName: (_d = itin.supplierName) !== null && _d !== void 0 ? _d : null,
                                supplierContact: (_e = itin.supplierContact) !== null && _e !== void 0 ? _e : null,
                                totalPriceAfterMarkup: (_f = itin.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                priceDetailsList: {
                                    create: ((_g = itin.priceDetails) !== null && _g !== void 0 ? _g : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f;
                                        return ({
                                            category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                            price: (_b = pd.price) !== null && _b !== void 0 ? _b : null,
                                            priceAfterMarkup: (_c = pd.priceAfterMarkup) !== null && _c !== void 0 ? _c : null,
                                            quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                            totalPrice: (_e = pd.totalPrice) !== null && _e !== void 0 ? _e : null,
                                            totalPriceAfterMarkup: (_f = pd.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                        });
                                    }),
                                },
                                guideId: (_h = itin.guideId) !== null && _h !== void 0 ? _h : null,
                            });
                        }),
                    },
                    tours: {
                        connect: ((_m = data.tours) !== null && _m !== void 0 ? _m : []).map((tourId) => ({
                            id: tourId,
                        })),
                    },
                },
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            const user = await prisma.user.findUnique({
                where: { id: userid },
                select: { name: true },
            });
            const userName = (user === null || user === void 0 ? void 0 : user.name) || "Unknown User";
            const lead = await prisma.lead.findUnique({
                where: { leadNo: data.leadRefNo },
            });
            if (lead) {
                const newStage = "quotation_created"; // Define the new stage value
                await prisma.lead.update({
                    where: { id: lead.id },
                    data: { stage: newStage },
                });
                await prisma.followUp.create({
                    data: {
                        leadId: lead.id,
                        oldStage: lead.stage,
                        newStage: newStage,
                        notes: "Quotation created",
                        userName: userName !== null && userName !== void 0 ? userName : null,
                    },
                });
            }
            // ✅ Send notification
            await this.notificationService.notifyAdmin({
                type: "quotation_created",
                title: "New Quotation Created",
                message: `New quotation created by ${userName}.\nLead Reference No: ${data.leadRefNo || "N/A"}\nGuest Name: ${data.guestName || "Guest Name"}.`,
                entityType: "quotation",
                entityId: quotation.id,
                // Change as needed
            });
            console.log("Quotation created in repo: ******************", quotation.refNo);
            return quotation;
        }
        catch (error) {
            console.error("Database error while creating quotation:", error);
            throw new Error("Failed to create quotation. Please try again later.");
        }
    }
    // --- NEW DUPLICATION METHOD ---
    async duplicateQuotation(id, userid) {
        try {
            const originalQuotation = await prisma.quotation.findUnique({
                where: { id, userid: userid }, // Added userid filter
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true, // Make sure to include tours for copying
                },
            });
            if (!originalQuotation) {
                throw new Error("Original quotation not found for duplication or user doesn't own it.");
            }
            // 1. Get all refNos for this lead to determine the next duplicate number
            const allQuotationsForLead = await prisma.quotation.findMany({
                where: { leadRefNo: originalQuotation.leadRefNo, userid: userid }, // Added userid filter
                select: { refNo: true },
            });
            const existingRefNosForLead = allQuotationsForLead.map((q) => q.refNo);
            // The base for duplication should be the original refNo without any -D suffix,
            // as new duplicates will be appended to this base.
            const baseRefForDuplication = originalQuotation.refNo.replace(/-D\d+$/, "");
            const newRefNo = this.getDuplicateRefNo(baseRefForDuplication, existingRefNosForLead);
            // 2. Prepare data for the new quotation
            // Dates from Prisma are already Date objects, so parseDate needs adjustment for this context
            const parseDate = (d) => (d ? d : null);
            const duplicateData = {
                refNo: newRefNo,
                leadRefNo: originalQuotation.leadRefNo,
                leadAgent: originalQuotation.leadAgent,
                status: client_1.QuotationStatus.Draft, // Set duplicated quotation status to 'Draft'
                guestName: originalQuotation.guestName,
                remarks: `Duplicate of ${originalQuotation.refNo} (created by user ${userid} at ${new Date().toISOString()})`, // Add a remark
                currency: originalQuotation.currency,
                totalBeforeMarkup: originalQuotation.totalBeforeMarkup,
                totalAfterMarkup: originalQuotation.totalAfterMarkup,
                travelDates: originalQuotation.travelDates,
                userid: userid,
                agentId: originalQuotation.agentId,
                // createdAt and updatedAt will be automatically set by Prisma for the new record
            };
            // Handle nested relations for creation
            duplicateData.accommodations = {
                create: originalQuotation.accommodations.map((acc) => ({
                    hotel: acc.hotel,
                    hotelAddress: acc.hotelAddress,
                    supplierId: acc.supplierId,
                    supplierName: acc.supplierName,
                    supplierContact: acc.supplierContact,
                    room: acc.room,
                    travelDate: parseDate(acc.travelDate),
                    endDate: parseDate(acc.endDate),
                    nights: acc.nights,
                    surchargeNights: acc.surchargeNights,
                    quantity: acc.quantity,
                    totalPrice: acc.totalPrice,
                    totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
                    priceDetailsAccommodation: {
                        create: acc.priceDetailsAccommodation.map((pd) => ({
                            roomId: pd.roomId,
                            category: pd.category,
                            pax: pd.pax,
                            price: pd.price,
                            priceAfterMarkup: pd.priceAfterMarkup,
                            costPrice: pd.costPrice,
                            optionalPrice: pd.optionalPrice,
                            quantity: pd.quantity,
                            totalPrice: pd.totalPrice,
                            totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                        })),
                    },
                })),
            };
            duplicateData.itineraries = {
                create: originalQuotation.itineraries.map((itin) => ({
                    date: parseDate(itin.date),
                    label: itin.label,
                    category: itin.category,
                    supplierId: itin.supplierId,
                    supplierName: itin.supplierName,
                    supplierContact: itin.supplierContact,
                    totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
                    priceDetailsList: {
                        create: itin.priceDetailsList.map((pd) => ({
                            category: pd.category,
                            price: pd.price,
                            priceAfterMarkup: pd.priceAfterMarkup,
                            quantity: pd.quantity,
                            totalPrice: pd.totalPrice,
                            totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                        })),
                    },
                    guideId: itin.guideId,
                })),
            };
            // For many-to-many relationships (like tours), use `connect` to link existing tours
            duplicateData.tours = {
                connect: originalQuotation.tours.map((tour) => ({ id: tour.id })),
            };
            const duplicatedQuotation = await prisma.quotation.create({
                data: duplicateData,
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            console.log("Quotation duplicated in repo:", duplicatedQuotation.refNo);
            return duplicatedQuotation;
        }
        catch (error) {
            if (error instanceof Error) {
                console.error("Error duplicating quotation:", error.message);
            }
            else {
                console.error("Error duplicating quotation:", error);
            }
            throw new Error("Failed to duplicate quotation");
        }
    }
    async getQuotationByLeadRefNo(leadRefNo) {
        try {
            const quotation = await prisma.quotation.findMany({
                where: { leadRefNo }, // Added userid filter
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true }
                    },
                    tours: true,
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                            User: true, // Include user details
                        },
                    },
                },
            });
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            console.log("Quotation in repo", quotation);
            return quotation;
        }
        catch (error) {
            console.error("Error fetching quotation:", error);
            throw error;
        }
    }
    async getQuotationById(id, userid) {
        try {
            const quotation = await prisma.quotation.findUnique({
                where: { id, userid: userid }, // Added userid filter
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true,
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                        },
                    },
                },
            });
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            console.log("Quotation in repo", quotation);
            return quotation;
        }
        catch (error) {
            console.error("Error fetching quotation:", error);
            throw error;
        }
    }
    async getAllQuotations(userid) {
        try {
            const quotations = await prisma.quotation.findMany({
                where: { userid: userid }, // Added userid filter
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true }, // Fix incorrect reference
                    },
                    tours: true,
                },
            });
            return quotations;
        }
        catch (error) {
            console.error("Error fetching quotations:", error);
            throw new Error("Could not fetch quotations");
        }
    }
    async updateQuotation(id, data, userid) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        try {
            const parseDate = (d) => d ? new Date(d) : null;
            // Delete nested children first.  Using `deleteMany` with relations is generally more efficient.
            await prisma.priceDetailsAccommodation.deleteMany({
                where: { accommodation: { quotationId: id } },
            });
            await prisma.accommodation.deleteMany({ where: { quotationId: id } });
            await prisma.priceDetailItinerary.deleteMany({
                where: { itinerary: { quotationId: id } },
            });
            await prisma.itinerary.deleteMany({ where: { quotationId: id } });
            // Disconnect tours before updating.  Important for proper updates
            await prisma.quotation.update({
                where: { id, userid: userid },
                data: {
                    tours: {
                        set: [],
                    },
                },
            });
            const updateData = {
                leadRefNo: data.leadRefNo,
                leadAgent: (_a = data.leadAgent) !== null && _a !== void 0 ? _a : null,
                status: (_b = data.status) !== null && _b !== void 0 ? _b : null,
                guestName: (_c = data.guestName) !== null && _c !== void 0 ? _c : null,
                remarks: (_d = data.remarks) !== null && _d !== void 0 ? _d : null,
                currency: (_e = data.currency) !== null && _e !== void 0 ? _e : null,
                totalBeforeMarkup: (_f = data.totalBeforeMarkup) !== null && _f !== void 0 ? _f : null,
                totalAfterMarkup: (_g = data.totalAfterMarkup) !== null && _g !== void 0 ? _g : null,
                travelDates: data.travelDates,
                agentId: (_h = data.agentId) !== null && _h !== void 0 ? _h : null,
            };
            if (data.accommodations) {
                updateData.accommodations = {
                    create: data.accommodations.map((acc) => {
                        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
                        return ({
                            hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                            hotelAddress: (_b = acc.hotelAddress) !== null && _b !== void 0 ? _b : null,
                            supplierId: (_c = acc.supplierId) !== null && _c !== void 0 ? _c : null,
                            supplierName: (_d = acc.supplierName) !== null && _d !== void 0 ? _d : null,
                            supplierContact: (_e = acc.supplierContact) !== null && _e !== void 0 ? _e : null,
                            room: (_f = acc.room) !== null && _f !== void 0 ? _f : null,
                            travelDate: parseDate(acc.travelDate),
                            endDate: parseDate(acc.endDate),
                            nights: (_g = acc.nights) !== null && _g !== void 0 ? _g : null,
                            surchargeNights: (_h = acc.surchargeNights) !== null && _h !== void 0 ? _h : null,
                            quantity: (_j = acc.quantity) !== null && _j !== void 0 ? _j : null,
                            totalPrice: (_k = acc.totalPrice) !== null && _k !== void 0 ? _k : null,
                            totalPriceAfterMarkup: (_l = acc.totalPriceAfterMarkup) !== null && _l !== void 0 ? _l : null,
                            priceDetailsAccommodation: {
                                create: ((_m = acc.priceDetails) !== null && _m !== void 0 ? _m : []).map((pd) => {
                                    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                                    return ({
                                        roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                        category: (_b = pd.category) !== null && _b !== void 0 ? _b : null,
                                        pax: (_c = pd.pax) !== null && _c !== void 0 ? _c : null,
                                        price: (_d = pd.price) !== null && _d !== void 0 ? _d : null,
                                        priceAfterMarkup: (_e = pd.priceAfterMarkup) !== null && _e !== void 0 ? _e : null,
                                        costPrice: (_f = pd.costPrice) !== null && _f !== void 0 ? _f : null,
                                        optionalPrice: (_g = pd.optionalPrice) !== null && _g !== void 0 ? _g : null,
                                        quantity: (_h = pd.quantity) !== null && _h !== void 0 ? _h : null,
                                        totalPrice: (_j = pd.totalPrice) !== null && _j !== void 0 ? _j : null,
                                        totalPriceAfterMarkup: (_k = pd.totalPriceAfterMarkup) !== null && _k !== void 0 ? _k : null,
                                    });
                                }),
                            },
                        });
                    }),
                };
            }
            if (data.itineraries) {
                updateData.itineraries = {
                    create: data.itineraries.map((itin) => {
                        var _a, _b, _c, _d, _e, _f, _g, _h;
                        return ({
                            date: parseDate(itin.date),
                            label: (_a = itin.label) !== null && _a !== void 0 ? _a : null,
                            category: (_b = itin.category) !== null && _b !== void 0 ? _b : null,
                            supplierId: (_c = itin.supplierId) !== null && _c !== void 0 ? _c : null,
                            supplierName: (_d = itin.supplierName) !== null && _d !== void 0 ? _d : null,
                            supplierContact: (_e = itin.supplierContact) !== null && _e !== void 0 ? _e : null,
                            totalPriceAfterMarkup: (_f = itin.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                            priceDetailsList: {
                                create: ((_g = itin.priceDetails) !== null && _g !== void 0 ? _g : []).map((pd) => {
                                    var _a, _b, _c, _d, _e, _f;
                                    return ({
                                        category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                        price: (_b = pd.price) !== null && _b !== void 0 ? _b : null,
                                        priceAfterMarkup: (_c = pd.priceAfterMarkup) !== null && _c !== void 0 ? _c : null,
                                        quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                        totalPrice: (_e = pd.totalPrice) !== null && _e !== void 0 ? _e : null,
                                        totalPriceAfterMarkup: (_f = pd.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                    });
                                }),
                            },
                            guideId: (_h = itin.guideId) !== null && _h !== void 0 ? _h : null,
                        });
                    }),
                };
            }
            if (data.tours) {
                updateData.tours = {
                    connect: data.tours.map((tourId) => ({ id: tourId })),
                };
            }
            const updatedQuotation = await prisma.quotation.update({
                where: { id, userid: userid },
                data: updateData,
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            // Simulate the admin's updateQuotationStatus behavior for 'Accepted' status
            if (data.status === 'Accepted') { // Use data.status instead of updatedQuotation.status as it may not be updated yet
                const leadRefNo = updatedQuotation.leadRefNo;
                if (leadRefNo) {
                    const lead = await prisma.lead.findUnique({
                        where: { leadNo: leadRefNo },
                    });
                    if (lead) { // Check if the lead exists
                        await prisma.lead.update({
                            where: { leadNo: leadRefNo },
                            data: {
                                stage: 'quotation_accepted',
                                followUps: {
                                    create: {
                                        oldStage: lead.stage, // Use the fetched lead's stage
                                        newStage: 'quotation_accepted',
                                        notes: `Quotation ${updatedQuotation.refNo} accepted.`, // Access refNo after quotation update
                                        userName: userid, // Or the actual user name
                                    },
                                },
                            },
                        });
                    }
                    else {
                        console.warn(`Lead with leadNo ${leadRefNo} not found.  Skipping lead update.`);
                    }
                }
                else {
                    console.warn('leadRefNo is null or undefined.  Skipping lead update.');
                }
            }
            console.log("Updated Quotation", updatedQuotation);
            return updatedQuotation;
        }
        catch (error) {
            console.error("Error updating quotation:", error);
            throw new Error("Failed to update quotation");
        }
    }
    async deleteQuotation(id, userid) {
        console.log("ID in repository:", id); // Debugging
        const existingQuotation = await prisma.quotation.findUnique({
            where: { id, userid: userid }, // Added userid filter
        });
        if (!existingQuotation) {
            throw new Error("Quotation not found");
        }
        return await prisma.quotation.delete({
            where: { id },
        });
    }
    async updateQuotationStatus(id, status, userid) {
        try {
            // Find the user's name (optional - for follow-up notes)
            const user = await prisma.user.findUnique({
                where: { id: userid },
                select: { name: true }
            });
            const userName = (user === null || user === void 0 ? void 0 : user.name) || "Unknown User";
            const updatedQuotation = await prisma.quotation.update({
                where: { id, userid: userid }, // <--- ADDED userid to where clause
                data: { status },
                include: {
                    lead: true, // so we can access lead data
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true,
                },
            });
            if (!updatedQuotation)
                throw new Error("Quotation not found");
            // If status is 'Accepted', also update Lead and create FollowUp
            if (status === 'Accepted') {
                const leadUpdate = await prisma.lead.update({
                    where: { leadNo: updatedQuotation.leadRefNo },
                    data: {
                        stage: 'quotation_accepted',
                        followUps: {
                            create: {
                                oldStage: updatedQuotation.lead.stage,
                                newStage: 'quotation_accepted',
                                notes: `Quotation ${updatedQuotation.refNo} accepted by ${userName}.`, // Use the actual username here
                                userName: userName, // Use the actual username here
                            },
                        },
                    },
                });
            }
            await this.notificationService.notifyAdmin({
                type: "quotation_accepted",
                title: "Quotation Accepted",
                message: `Quotation ${updatedQuotation.refNo} has been accepted by ${userName}.`,
                entityType: "quotation",
                entityId: updatedQuotation.id,
            });
            return updatedQuotation;
        }
        catch (error) {
            console.error("Error updating quotation status:", error);
            throw error;
        }
    }
    async getAcceptedQuotationsByLeadId(leadId, userid) {
        try {
            const lead = await prisma.lead.findUnique({
                where: { id: leadId },
                include: {
                    Quotation: {
                        where: { status: 'Accepted', userid: userid }, // Added userid filter
                        include: {
                            accommodations: true,
                            itineraries: true,
                            tours: true,
                        },
                    },
                },
            });
            if (!lead)
                throw new Error("Lead not found");
            return lead;
        }
        catch (error) {
            console.error("Error fetching accepted quotations:", error);
            throw error;
        }
    }
}
exports.QuotationRepository = QuotationRepository;
//# sourceMappingURL=userquotationRepository.js.map