"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserLeadRepository = void 0;
const notificationService_1 = require("../../services/common/notificationService");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class UserLeadRepository {
    constructor() {
        this.notificationService = new notificationService_1.NotificationService(); // ✅ Initialize once
    }
    formatLeadDates(lead) {
        return {
            ...lead,
            travelDateFrom: lead.travelDateFrom
                ? new Date(lead.travelDateFrom).toISOString()
                : null,
            travelDateTo: lead.travelDateTo
                ? new Date(lead.travelDateTo).toISOString()
                : null,
        };
    }
    async createLead(leadData) {
        try {
            if (!leadData.empCode) {
                throw new Error('Employee code is missing or invalid');
            }
            console.log("Creating lead with data:", leadData.empCode);
            const user = await prismaClient_1.default.user.findUnique({
                where: { empCode: leadData.empCode },
                select: { name: true },
            });
            console.log("User found:", user);
            const userName = (user === null || user === void 0 ? void 0 : user.name) || "Unknown User";
            const newLead = await prismaClient_1.default.lead.create({
                data: {
                    ...leadData,
                    followUps: {
                        create: {
                            oldStage: 'new_lead',
                            newStage: 'new_lead',
                            notes: 'Lead created',
                            userName: userName, // or get from context
                        }
                    }
                },
                include: {
                    followUps: true,
                }
            });
            // 📦 Save notification to DB
            await this.notificationService.notifyAdmin({
                type: "lead_created",
                title: "New Lead Created",
                message: `A new lead was created by the user ${userName}`,
                entityType: "lead",
                entityId: newLead.id,
            });
            return this.formatLeadDates(newLead);
        }
        catch (error) {
            console.error("Error creating lead in user lead repository:", error);
            throw error;
        }
    }
    async getUserLeadsByUserId(userId) {
        try {
            // Step 1: Fetch empCode using userId
            const user = await prismaClient_1.default.user.findUnique({
                where: { id: userId },
                select: { empCode: true },
            });
            if (!user || !user.empCode) {
                throw new Error("User not found or empCode is missing");
            }
            // Step 2: Fetch leads using empCode
            const leads = await prismaClient_1.default.lead.findMany({
                where: {
                    empCode: user.empCode,
                },
                include: {
                    followUps: true,
                    Agent: true,
                    User: true
                }
            });
            return leads.map(this.formatLeadDates);
        }
        catch (error) {
            console.error("Error getting user leads in repository:", error);
            throw error;
        }
    }
    async getLeadById(leadId) {
        try {
            const lead = await prismaClient_1.default.lead.findUnique({
                where: {
                    id: leadId,
                },
                include: {
                    followUps: true,
                    Agent: true,
                    User: true
                }
            });
            return lead ? this.formatLeadDates(lead) : null;
        }
        catch (error) {
            console.error("Error getting lead by ID in user lead repository:", error);
            throw error;
        }
    }
    async updateLead(leadId, leadData) {
        try {
            // 🔍 Step 1: Find the lead by ID
            const lead = await prismaClient_1.default.lead.findUnique({
                where: { id: leadId },
            });
            console.log("Lead found:", lead);
            // ✅ Step 3: Proceed to update
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id: leadId },
                data: leadData,
            });
            return updatedLead ? this.formatLeadDates(updatedLead) : null;
        }
        catch (error) {
            console.error("Error updating lead in user lead repository:", error);
            return null;
        }
    }
    async deleteLead(leadId) {
        try {
            await prismaClient_1.default.lead.delete({
                where: {
                    id: leadId,
                },
            });
        }
        catch (error) {
            console.error("Error deleting lead in user lead repository:", error);
            throw error;
        }
    }
    async updateLeadStatus(id, status) {
        try {
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: { status },
                include: {
                    followUps: true
                }
            });
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead status:", error);
            return null;
        }
    }
    async updateLeadStage(id, newStage, userName, notes) {
        try {
            // Get existing lead to fetch oldStage
            const existingLead = await prismaClient_1.default.lead.findUnique({
                where: { id },
                select: { stage: true },
            });
            if (!existingLead)
                return null;
            // Optional: Validate userId exists in DB to avoid FK violation
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: {
                    stage: newStage,
                    followUps: {
                        create: {
                            oldStage: existingLead.stage,
                            newStage: newStage,
                            userName: userName,
                            notes: notes || '',
                        },
                    },
                },
                include: { followUps: true },
            });
            console.log(updatedLead);
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead stage:", error);
            return null;
        }
    }
}
exports.UserLeadRepository = UserLeadRepository;
//# sourceMappingURL=userLeadRepository.js.map