"use strict";
// backend/src/repositories/agent/agentRepository.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentRepository = void 0;
const client_1 = require("@prisma/client");
// Make sure this import path is correct for your interface file
const nanoid_1 = require("nanoid");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class AgentRepository {
    async getAgentByEmail(email) {
        try {
            return await prismaClient_1.default.agent.findUnique({
                where: { email: email },
            });
        }
        catch (error) {
            console.error("Error finding agent by email:", error);
            return null;
        }
    }
    async getAgentByPhone(phone) {
        try {
            return await prismaClient_1.default.agent.findUnique({
                where: { phone: phone },
            });
        }
        catch (error) {
            console.error("Error finding agent by phone:", error);
            return null;
        }
    }
    async getAgentByAgentCode(agentCode) {
        try {
            return await prismaClient_1.default.agent.findUnique({
                where: { agentCode: agentCode },
            });
        }
        catch (error) {
            console.error("Error finding agent by agent code:", error);
            return null;
        }
    }
    async createAgent(agentData, credentialsData) {
        var _a;
        try {
            // ✅ Create new agent
            const newAgent = await prismaClient_1.default.agent.create({
                data: {
                    agentCode: agentData.agentCode,
                    companyName: agentData.companyName,
                    contactPerson: agentData.contactPerson,
                    designation: agentData.designation,
                    companyAddress: agentData.companyAddress,
                    state: agentData.state,
                    email: agentData.email,
                    phone: agentData.phone,
                    markup: agentData.markup,
                    agentPrivilege: agentData.agentPrivilege,
                    deleted: (_a = agentData.deleted) !== null && _a !== void 0 ? _a : false,
                    credentials: (credentialsData === null || credentialsData === void 0 ? void 0 : credentialsData.username)
                        ? {
                            create: {
                                id: credentialsData.id || (0, nanoid_1.nanoid)(),
                                username: credentialsData.username,
                                password: credentialsData.password,
                                filename: credentialsData.filename || null,
                            },
                        }
                        : undefined,
                },
                include: { credentials: true },
            });
            return {
                ...newAgent,
                agentInfo: agentData.agentInfo || null,
            };
        }
        catch (error) {
            console.error("❌ Error creating agent in repository:", error);
            throw error;
        }
    }
    async updateAgent(agentId, agentData, credentialsData) {
        try {
            const existingAgent = await prismaClient_1.default.agent.findUnique({
                where: { id: agentId },
                select: {
                    credentialsId: true,
                    email: true,
                    agentCode: true,
                    phone: true,
                },
            });
            if (!existingAgent) {
                console.warn(`Agent with ID ${agentId} not found for update.`);
                return null;
            }
            // ✅ Step 2: Construct update payload
            const updatePayload = {
                agentCode: agentData.agentCode,
                companyName: agentData.companyName,
                contactPerson: agentData.contactPerson,
                designation: agentData.designation,
                companyAddress: agentData.companyAddress,
                state: agentData.state,
                email: agentData.email,
                phone: agentData.phone,
                markup: agentData.markup,
                agentPrivilege: agentData.agentPrivilege,
                deleted: agentData.deleted,
            };
            if (credentialsData) {
                if (existingAgent.credentialsId) {
                    const credentialUpdatePayload = {};
                    if (credentialsData.username !== undefined) {
                        credentialUpdatePayload.username = credentialsData.username;
                    }
                    if (credentialsData.password !== undefined) {
                        credentialUpdatePayload.password = credentialsData.password;
                    }
                    credentialUpdatePayload.filename = credentialsData.filename;
                    updatePayload.credentials = {
                        update: credentialUpdatePayload
                    };
                }
                else {
                    updatePayload.credentials = {
                        create: {
                            id: credentialsData.id || (0, nanoid_1.nanoid)(),
                            username: credentialsData.username,
                            password: credentialsData.password,
                            filename: credentialsData.filename,
                        },
                    };
                }
            }
            // ✅ Step 3: Perform update
            const updatedAgent = await prismaClient_1.default.agent.update({
                where: { id: agentId },
                data: updatePayload,
                include: { credentials: true },
            });
            return {
                ...updatedAgent,
                agentInfo: agentData.agentInfo || null,
            };
        }
        catch (error) {
            console.error("Error updating agent in repository:", error);
            return null;
        }
    }
    async getAllAgents() {
        try {
            const agents = await prismaClient_1.default.agent.findMany({
                // Only fetch agents not marked as deleted
                // where: { deleted: false }, // Alternative if null is not used
                include: { credentials: true },
                orderBy: { createdAt: "desc" },
            });
            return agents.map((agent) => ({
                ...agent,
                email: agent.email,
                agentInfo: null, // Ensure agentInfo is included
            }));
        }
        catch (error) {
            console.error("Error getting agents in repository:", error);
            throw error;
        }
    }
    async deleteAgent(agentId) {
        try {
            const deletedAgent = await prismaClient_1.default.agent.delete({
                where: { id: agentId },
            });
            console.log(`  Agent ${agentId} and associated credentials deleted successfully.`);
            return {
                ...deletedAgent,
                agentInfo: null, // Ensure agentInfo is included
            };
        }
        catch (error) {
            console.error("Error deleting agent in repository:", error);
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2025") {
                console.warn(`Agent with ID ${agentId} not found for deletion.`);
                return null;
            }
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError &&
                error.code === "P2003") {
                console.error(`  Cannot delete agent ${agentId} due to existing related records (e.g., Leads).`);
                throw new Error(`   Cannot delete agent: Agent has related records (like Leads). Consider deactivating instead.`);
            }
            throw error;
        }
    }
    async softDeleteAgent(agentId) {
        if (!agentId) {
            throw new Error("Agent ID is required for soft deletion.");
        }
        const updatedAgent = await prismaClient_1.default.agent.update({
            where: { id: agentId },
            data: { deleted: true },
            include: { credentials: true },
        });
        return {
            ...updatedAgent,
            agentInfo: null, // Ensure agentInfo is included
        };
    }
    async restoreAgent(agentId) {
        try {
            const updatedAgent = await prismaClient_1.default.agent.update({
                where: { id: agentId },
                data: { deleted: false }, // Set 'deleted' to false
                include: { credentials: true },
            });
            return {
                ...updatedAgent,
                agentInfo: null,
            };
        }
        catch (error) {
            console.error("Error restoring agent in repository:", error);
            return null;
        }
    }
}
exports.AgentRepository = AgentRepository;
//# sourceMappingURL=userAgentRepository.js.map