"use strict";
// In OperationDashboardRepository.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationDashboardRepository = void 0;
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const client_1 = require("@prisma/client");
const date_fns_1 = require("date-fns");
class OperationDashboardRepository {
    async getDashboardData(userId) {
        try {
            const assignedBookingsCount = await prismaClient_1.default.booking.count({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                }
            });
            // Calculate total **quote price** (total Revenue)
            const totalQuotePriceAggregate = await prismaClient_1.default.booking.aggregate({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                _sum: {
                    quotePrice: true,
                },
            });
            const totalQuotePrice = totalQuotePriceAggregate._sum.quotePrice || 0;
            // Calculate total **actual cost** (total Cost) from booking
            const totalActualCostAggregate = await prismaClient_1.default.booking.aggregate({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                _sum: {
                    actualCost: true,
                },
            });
            const totalActualCost = totalActualCostAggregate._sum.actualCost || 0;
            // Calculate total **actual price** based on accommodations and itineraries
            let totalActualAccomodationPrice = 0;
            let totalActualItineraryPrice = 0;
            const bookings = await prismaClient_1.default.booking.findMany({
                where: {
                    lead: {
                        assignedOperationId: userId,
                    },
                },
                include: {
                    accommodations: true,
                    itineraries: true,
                },
            });
            bookings.forEach(booking => {
                booking.accommodations.forEach(accommodation => {
                    if (accommodation.actualPrice) {
                        totalActualAccomodationPrice += accommodation.actualPrice.toNumber();
                    }
                });
                booking.itineraries.forEach(itinerary => {
                    if (itinerary.actualPrice) {
                        totalActualItineraryPrice += itinerary.actualPrice.toNumber();
                    }
                });
            });
            const totalActualOverallPrice = totalActualAccomodationPrice + totalActualItineraryPrice;
            // Calculate total profit (quotePrice - actualCost)
            const totalProfit = totalQuotePrice - totalActualCost;
            const latestBooking = await prismaClient_1.default.booking.findFirst({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                orderBy: { updatedAt: 'desc' },
                include: { lead: true }
            });
            return {
                totalBookings: assignedBookingsCount,
                totalQuotePrice: totalQuotePrice, // Total Revenue (quotePrice)
                totalActualCost: totalActualCost, // Total Cost (actualCost)
                totalProfit: totalProfit, // Total Profit (quotePrice - actualCost)
                latestBooking,
                totalActualOverallPrice
            };
        }
        catch (error) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve Operation dashboard data");
        }
    }
    async getTopBookingsByProfit(userId, limit = 5) {
        try {
            const topBookings = await prismaClient_1.default.booking.findMany({
                where: {
                    lead: {
                        assignedOperationId: userId, // Only bookings assigned to the user
                    }
                },
                orderBy: {
                    profit: 'desc'
                },
                take: limit,
                select: {
                    id: true,
                    reference: true,
                    actualCost: true,
                    quotePrice: true,
                    profit: true,
                    lead: {
                        select: {
                            agentName: true,
                            id: true,
                            assignedOperationId: true
                        }
                    }
                },
            });
            // Calculate profit for each booking (if not already correct)
            const bookingsWithCalculatedProfit = topBookings.map(booking => ({
                ...booking,
                profit: booking.quotePrice - booking.actualCost, // Calculate profit
            }));
            return bookingsWithCalculatedProfit;
        }
        catch (error) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve Top 5 Bookings");
        }
    }
    async getBookingLeadOverview(timeframe = "weekly", userId) {
        try {
            let startDate;
            let dateFormat;
            let endDate = new Date(); // Default end date is now
            const today = new Date(); // Use today as the reference point
            switch (timeframe) {
                case 'today':
                    startDate = (0, date_fns_1.startOfDay)(today); //start of the day
                    endDate = (0, date_fns_1.endOfDay)(today); // end of the day
                    dateFormat = 'HH:mm'; //Format time
                    break;
                case 'weekly':
                    startDate = (0, date_fns_1.subWeeks)(today, 1);
                    dateFormat = 'yyyy-MM-dd';
                    break;
                case 'monthly':
                    startDate = (0, date_fns_1.subMonths)(today, 1);
                    dateFormat = 'yyyy-MM';
                    break;
                case 'yearly':
                    startDate = (0, date_fns_1.subYears)(today, 1);
                    dateFormat = 'yyyy';
                    break;
                default:
                    startDate = (0, date_fns_1.subWeeks)(today, 1);
                    dateFormat = 'yyyy-MM-dd';
                    break;
            }
            // Fetch bookings instead of leads, and group by lead ID
            const bookings = await prismaClient_1.default.booking.findMany({
                where: {
                    createdAt: {
                        gte: startDate,
                        lte: endDate,
                    },
                    lead: {
                        assignedOperationId: userId, // Only bookings related to leads assigned to the user
                    },
                },
                select: {
                    createdAt: true,
                    lead: {
                        select: {
                            id: true, // Lead ID for de-duplication
                            status: true, //Lead Status
                        }
                    },
                    // Add other booking fields you might need for categorization/analysis here
                },
            });
            const leadData = {};
            bookings.forEach(booking => {
                if (!booking.lead)
                    return; // Skip bookings without associated leads (shouldn't happen but good to check)
                const leadId = booking.lead.id;
                const date = (0, date_fns_1.format)(booking.createdAt, dateFormat);
                if (!leadData[date]) {
                    leadData[date] = {
                        leads: new Set(), // Use a Set to track unique lead IDs
                        booked: 0,
                        cancelled: 0,
                    };
                }
                leadData[date].leads.add(leadId); // Add the lead ID to the set
                if (booking.lead.status === client_1.LeadStatus.active) {
                    leadData[date].booked++;
                }
                else if (booking.lead.status === client_1.LeadStatus.closed) {
                    leadData[date].cancelled++;
                }
            });
            const chartData = Object.entries(leadData).map(([date, data]) => ({
                date,
                leads: data.leads.size, // Count the unique lead IDs in the set
                booked: data.booked,
                cancelled: data.cancelled,
            }));
            return chartData;
        }
        catch (error) {
            console.error("Error fetching booking leads data:", error);
            throw new Error("Failed to retrieve booking leads data");
        }
    }
}
exports.OperationDashboardRepository = OperationDashboardRepository;
//# sourceMappingURL=operationDashboardRepository.js.map