"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BookingRepository = void 0;
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class BookingRepository {
    async fetchQuotationWithDetails(quotationId) {
        try {
            return await prismaClient_1.default.quotation.findUnique({
                where: { id: quotationId },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                },
            });
        }
        catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }
    async handleCreateBooking(quotation) {
        var _a, _b;
        try {
            console.log("Quotation in BookingRepository:", quotation);
            if (!quotation)
                throw new Error("Quotation not found");
            const booking = await prismaClient_1.default.booking.create({
                data: {
                    reference: `BK-${quotation.refNo}`,
                    leadNo: quotation.leadRefNo,
                    quotationId: quotation.id,
                    travelDates: quotation.travelDates,
                    agent: quotation.leadAgent,
                    quotePrice: Number(quotation.totalAfterMarkup || 0),
                    actualCost: quotation.totalBeforeMarkup || 0,
                    actualPrice: 0,
                    profit: 0,
                    accommodations: {
                        create: ((_a = quotation.accommodations) !== null && _a !== void 0 ? _a : []).map((acc) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h;
                            return ({
                                hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                                room: (_b = acc.room) !== null && _b !== void 0 ? _b : null,
                                travelDate: acc.travelDate ? new Date(acc.travelDate) : null,
                                endDate: acc.endDate ? new Date(acc.endDate) : null,
                                nights: (_c = acc.nights) !== null && _c !== void 0 ? _c : null,
                                surchargeNights: (_d = acc.surchargeNights) !== null && _d !== void 0 ? _d : null,
                                quantity: (_e = acc.quantity) !== null && _e !== void 0 ? _e : null,
                                totalPrice: (_f = acc.totalPrice) !== null && _f !== void 0 ? _f : null,
                                totalPriceAfterMarkup: (_g = acc.totalPriceAfterMarkup) !== null && _g !== void 0 ? _g : null,
                                quotationId: quotation.id,
                                priceDetailsAccommodation: {
                                    create: ((_h = acc.priceDetailsAccommodation) !== null && _h !== void 0 ? _h : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                                        return ({
                                            roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                            pax: (_b = pd.pax) !== null && _b !== void 0 ? _b : null,
                                            price: (_c = pd.price) !== null && _c !== void 0 ? _c : null,
                                            priceAfterMarkup: (_d = pd.priceAfterMarkup) !== null && _d !== void 0 ? _d : null,
                                            costPrice: (_e = pd.costPrice) !== null && _e !== void 0 ? _e : null,
                                            optionalPrice: (_f = pd.optionalPrice) !== null && _f !== void 0 ? _f : null,
                                            quantity: (_g = pd.quantity) !== null && _g !== void 0 ? _g : null,
                                            totalPrice: (_h = pd.totalPrice) !== null && _h !== void 0 ? _h : null,
                                            totalPriceAfterMarkup: (_j = pd.totalPriceAfterMarkup) !== null && _j !== void 0 ? _j : null,
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                    itineraries: {
                        create: ((_b = quotation.itineraries) !== null && _b !== void 0 ? _b : []).map((itin) => {
                            var _a, _b, _c, _d;
                            return ({
                                quotationId: quotation.id,
                                date: itin.date ? new Date(itin.date) : null,
                                label: (_a = itin.label) !== null && _a !== void 0 ? _a : null,
                                category: (_b = itin.category) !== null && _b !== void 0 ? _b : null,
                                guideId: (_c = itin.guideId) !== null && _c !== void 0 ? _c : null,
                                priceDetails: {}, // optionally populate if needed
                                priceDetailsList: {
                                    create: ((_d = itin.priceDetailsList) !== null && _d !== void 0 ? _d : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f;
                                        return ({
                                            category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                            price: (_b = pd.price) !== null && _b !== void 0 ? _b : null,
                                            priceAfterMarkup: (_c = pd.priceAfterMarkup) !== null && _c !== void 0 ? _c : null,
                                            quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                            totalPrice: (_e = pd.totalPrice) !== null && _e !== void 0 ? _e : null,
                                            totalPriceAfterMarkup: (_f = pd.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                },
                include: {
                    lead: true, // ✅ This includes the Lead details
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                },
            });
            console.log("Booking created successfully with lead details:", booking);
            return booking;
        }
        catch (error) {
            console.error("Error creating booking:", error);
            throw new Error("Failed to create booking from quotation");
        }
    }
    async getAllBookings(userId) {
        try {
            const whereClause = {};
            if (userId) {
                whereClause.lead = {
                    assignedOperationId: userId,
                };
            }
            return await prismaClient_1.default.booking.findMany({
                where: whereClause,
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                },
                orderBy: {
                    updatedAt: 'desc',
                },
            });
        }
        catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve bookings");
        }
    }
    async getBookingById(id) {
        try {
            const booking = await prismaClient_1.default.booking.findUnique({
                where: { id },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                        },
                    },
                },
            });
            if (!booking) {
                throw new Error("Booking not found");
            }
            console.log("Booking in repo", booking);
            return booking;
        }
        catch (error) {
            console.error("Error fetching Booking:", error);
            throw error; // Re-throw to handle in the calling function
        }
    }
    async findBookingWithAccommodations(bookingId) {
        return prismaClient_1.default.booking.findUnique({
            where: { id: bookingId },
            include: { accommodations: true },
        });
    }
    async updateAccommodation(accommodationId, updateData) {
        try {
            const existing = await prismaClient_1.default.bookingAccommodation.findUnique({
                where: { id: accommodationId },
            });
            if (!existing) {
                throw new Error("Accommodation not found");
            }
            // Update main accommodation data
            const updatedAccommodation = await prismaClient_1.default.bookingAccommodation.update({
                where: { id: accommodationId },
                data: {
                    hotel: updateData.hotel,
                    room: updateData.room,
                    supplierName: updateData.supplierName,
                    supplierContact: updateData.supplierContact,
                    travelDate: updateData.travelDate ? new Date(updateData.travelDate) : undefined,
                    nights: updateData.nights,
                    quotePrice: updateData.quotePrice,
                    actualPrice: updateData.actualPrice,
                    confirmationNo: updateData.confirmationNo,
                    details: updateData.details,
                    quantity: updateData.quantity,
                },
            });
            // Handle price details if provided
            if (updateData.priceDetailsAccommodation) {
                // First delete existing price details
                await prismaClient_1.default.bookingPriceDetailsAccommodation.deleteMany({
                    where: { accommodationId: accommodationId }
                });
                // Create new price details
                await prismaClient_1.default.bookingPriceDetailsAccommodation.createMany({
                    data: updateData.priceDetailsAccommodation.map(detail => ({
                        ...detail,
                        accommodationId: accommodationId,
                        id: undefined // Let database generate ID
                    }))
                });
            }
            return updatedAccommodation;
        }
        catch (error) {
            console.error("Error updating accommodation:", error);
            throw error;
        }
    }
    async updateItinerary(itineraryId, updateData) {
        try {
            const existing = await prismaClient_1.default.bookingItinerary.findUnique({
                where: { id: itineraryId },
                include: { priceDetailsList: true }
            });
            if (!existing) {
                throw new Error("Itinerary not found");
            }
            // Update main itinerary data
            const updatedItinerary = await prismaClient_1.default.bookingItinerary.update({
                where: { id: itineraryId },
                data: {
                    date: updateData.date ? new Date(updateData.date) : undefined,
                    label: updateData.label,
                    category: updateData.category,
                    supplierId: updateData.supplierId,
                    supplierName: updateData.supplierName,
                    supplierContact: updateData.supplierContact,
                    quotePrice: updateData.quotePrice,
                    actualPrice: updateData.actualPrice,
                    confirmationNo: updateData.confirmationNo,
                    details: updateData.details,
                },
            });
            // Handle price details if provided
            if (updateData.priceDetailsList) {
                // First delete existing price details
                await prismaClient_1.default.bookingPriceDetailItinerary.deleteMany({
                    where: { itineraryId: itineraryId }
                });
                // Create new price details
                await prismaClient_1.default.bookingPriceDetailItinerary.createMany({
                    data: updateData.priceDetailsList.map(detail => ({
                        ...detail,
                        itineraryId: itineraryId,
                        id: undefined // Let database generate ID
                    }))
                });
            }
            return updatedItinerary;
        }
        catch (error) {
            console.error("Error updating itinerary:", error);
            throw error;
        }
    }
}
exports.BookingRepository = BookingRepository;
;
//# sourceMappingURL=UserbookingRepository.js.map