"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTeamDashboardRepository = void 0;
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const date_fns_1 = require("date-fns");
function getLastNDates(n) {
    const dates = [];
    for (let i = n - 1; i >= 0; i--) {
        dates.push((0, date_fns_1.startOfDay)((0, date_fns_1.subDays)(new Date(), i)));
    }
    return dates;
}
class UserTeamDashboardRepository {
    async getCountsPerDay(teamId, leadPersonEmpCode, tableName, n = 7) {
        const dates = getLastNDates(n);
        const dateField = tableName === 'lead' ? 'createdOn' : 'createdAt';
        // Add empCode filter for 'lead' table. Also filter for teamId on users/agents.
        let whereClause = {
            [dateField]: {
                gte: dates[0],
            },
        };
        if (tableName === 'lead') {
            whereClause.empCode = leadPersonEmpCode; // Only leads created by this user.
        }
        else if (tableName === 'user') {
            whereClause.teamId = teamId;
        }
        const results = await prismaClient_1.default[tableName].findMany({
            where: whereClause,
            select: {
                [dateField]: true,
            },
        });
        const groupedResults = results.reduce((acc, curr) => {
            var _a;
            const dateKey = (_a = (curr.createdAt || curr.createdOn)) === null || _a === void 0 ? void 0 : _a.toDateString();
            if (dateKey) {
                acc[dateKey] = (acc[dateKey] || 0) + 1;
            }
            return acc;
        }, {});
        const counts = dates.map(date => {
            const found = results.find((r) => { var _a; return ((_a = (r.createdAt || r.createdOn)) === null || _a === void 0 ? void 0 : _a.toDateString()) === date.toDateString(); });
            return groupedResults[date.toDateString()] || 0;
        });
        return counts;
    }
    async getDashboardData(teamId, userId) {
        try {
            console.log(`getDashboardData: teamId=${teamId}, userId=${userId}`);
            // *** FETCH THE EMPCODE ***
            const user = await prismaClient_1.default.user.findUnique({
                where: { id: userId },
                select: { empCode: true, name: true }, // Include name in select
            });
            if (!user || !user.empCode) {
                console.warn(`User with ID ${userId} not found, or empCode is missing.`);
                return {
                    totalLeads: 0,
                    bookedLeads: 0,
                    newLeads: [],
                    agentCount: 0,
                    leadPersonName: null,
                    cancelledLeads: 0,
                    dataSeries: {
                        agents: [],
                        leads: [],
                        totalQuotationsCreated: 0,
                        bookings: [],
                    },
                }; // Return default data if user not found or no empCode
            }
            const leadPersonEmpCode = user.empCode;
            const leadPersonName = user.name; // Store user's name
            console.log(`Fetched empCode: ${leadPersonEmpCode} for userId: ${userId}`);
            const totalLeads = await prismaClient_1.default.lead.count({
                where: {
                    empCode: leadPersonEmpCode,
                },
            });
            //  **** THIS IS THE FIX - Adjust the WHERE clause as per your data model ****
            console.log(leadPersonEmpCode, "leadperson emp code");
            const totalQuotationsCreated = await prismaClient_1.default.quotation.count({
                where: {
                    lead: {
                        empCode: leadPersonEmpCode,
                    },
                },
            });
            console.log(totalQuotationsCreated, "total quotations created");
            // Revised bookedLeads Query (count leads directly)
            const bookedLeads = await prismaClient_1.default.lead.count({
                where: {
                    empCode: leadPersonEmpCode,
                    stage: "quotation_accepted", //  or "trip_started" or "trip_completed" depending on your def
                },
            });
            // Added Cancelled Leads Count
            const cancelledLeads = await prismaClient_1.default.lead.count({
                where: {
                    empCode: leadPersonEmpCode,
                    stage: "canceled", // Assuming you have a "canceled" stage
                },
            });
            const newLeads = await prismaClient_1.default.lead.findMany({
                where: {
                    empCode: leadPersonEmpCode, // Leads created by the specific lead person
                    stage: "new_lead",
                },
            });
            const agentCount = await prismaClient_1.default.agent.count(); // Assuming agents also have teamId
            const leadsData = await this.getCountsPerDay(teamId, leadPersonEmpCode, 'lead');
            const agentsData = await this.getCountsPerDay(teamId, leadPersonEmpCode, 'agent');
            const bookingsData = await prismaClient_1.default.lead.groupBy({
                by: ['createdOn'],
                _count: { _all: true },
                where: {
                    empCode: leadPersonEmpCode,
                    stage: 'quotation_accepted',
                    createdOn: {
                        gte: getLastNDates(7)[0],
                    },
                },
            });
            const dates = getLastNDates(7);
            const bookings = dates.map(date => {
                const found = bookingsData.find(r => r.createdOn.toDateString() === date.toDateString());
                return found ? found._count._all : 0;
            });
            return {
                totalLeads,
                bookedLeads,
                newLeads,
                totalQuotationsCreated,
                agentCount,
                leadPersonName, // Return user's name
                cancelledLeads,
                dataSeries: {
                    agents: agentsData,
                    leads: leadsData,
                    bookings: bookings,
                },
            };
        }
        catch (error) {
            console.error("Repository Error:", error);
            throw new Error("Failed to retrieve dashboard data");
        }
    }
}
exports.UserTeamDashboardRepository = UserTeamDashboardRepository;
//# sourceMappingURL=UserTeamDashboardRepository.js.map