"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TransferRepository = void 0;
const prismaClient_1 = __importDefault(require("../config/prismaClient"));
const nanoid_1 = require("nanoid");
// Enum to validate vehicle types
const validVehicles = [
    "MIDNIGHT_SURCHARGE",
    "ALPHARD_VELLFIRE",
    "EIGHTEEN_SEATER",
    "STAREX",
    "TEN_SEATER",
    "SEDAN",
];
class TransferRepository {
    createTransfer(data) {
        return __awaiter(this, void 0, void 0, function* () {
            // Validate vehicle type before inserting
            for (const price of data.prices) {
                if (!validVehicles.includes(price.vehicle)) {
                    throw new Error(`Invalid vehicle type: ${price.vehicle}`);
                }
            }
            return yield prismaClient_1.default.transfer.create({
                data: {
                    id: (0, nanoid_1.nanoid)(12),
                    name: data.name,
                    to: data.to,
                    from: data.from,
                    supplier: data.supplierId, // Use supplierId instead of supplier object
                    terms: data.terms,
                    Price: {
                        create: data.prices.map((price) => ({
                            id: (0, nanoid_1.nanoid)(12),
                            vehicle: price.vehicle,
                            pax: price.pax,
                            price: price.price,
                            optionalPrice: price.optionalPrice,
                            minPax: price.minPax,
                            maxPax: price.maxPax,
                            transferId: (0, nanoid_1.nanoid)(12),
                        })),
                    },
                },
                include: { Price: true },
            });
        });
    }
    getTransfers() {
        return __awaiter(this, void 0, void 0, function* () {
            return yield prismaClient_1.default.transfer.findMany({ include: { Price: true } });
        });
    }
    getTransferById(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield prismaClient_1.default.transfer.findUnique({ where: { id }, include: { Price: true } });
        });
    }
    updateTransfer(id, data) {
        return __awaiter(this, void 0, void 0, function* () {
            // Validate vehicle type before updating
            for (const price of data.prices) {
                if (!validVehicles.includes(price.vehicle)) {
                    throw new Error(`Invalid vehicle type: ${price.vehicle}`);
                }
            }
            return yield prismaClient_1.default.transfer.update({
                where: { id },
                data: {
                    name: data.name,
                    from: data.from,
                    to: data.to,
                    supplier: data.supplierId,
                    terms: data.terms,
                    Price: {
                        upsert: data.prices.map((price) => ({
                            where: { id: price.id || (0, nanoid_1.nanoid)(12) }, // Use existing ID or generate a new one
                            create: {
                                id: (0, nanoid_1.nanoid)(12),
                                vehicle: price.vehicle,
                                pax: price.pax,
                                price: price.price,
                                optionalPrice: price.optionalPrice,
                                minPax: price.minPax,
                                maxPax: price.maxPax,
                                transferId: id, // Associate with correct transfer ID
                            },
                            update: {
                                vehicle: price.vehicle,
                                pax: price.pax,
                                price: price.price,
                                optionalPrice: price.optionalPrice,
                                minPax: price.minPax,
                                maxPax: price.maxPax,
                            },
                        })),
                    },
                },
                include: { Price: true },
            });
        });
    }
    deleteTransfer(id) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield prismaClient_1.default.transfer.delete({ where: { id }, include: { Price: true } });
        });
    }
}
exports.TransferRepository = TransferRepository;
