"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentLeadRepository = void 0;
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const notificationService_1 = require("../../services/common/notificationService"); // Import NotificationService
class AgentLeadRepository {
    constructor() {
        this.notificationService = new notificationService_1.NotificationService(); // Initialize NotificationService
    }
    formatLeadDates(lead) {
        return {
            ...lead,
            travelDateFrom: lead.travelDateFrom
                ? new Date(lead.travelDateFrom).toISOString()
                : null,
            travelDateTo: lead.travelDateTo
                ? new Date(lead.travelDateTo).toISOString()
                : null,
            createdOn: lead.createdOn // Fixes issue with missing createdOn property
                ? new Date(lead.createdOn).toISOString() : null, // Apply same logic to other date fields
        };
    }
    async createLead(leadData) {
        try {
            const newLead = await prismaClient_1.default.lead.create({
                data: {
                    ...leadData,
                    followUps: {
                        create: {
                            oldStage: 'new_lead',
                            newStage: 'new_lead',
                            notes: 'Lead created',
                            userName: leadData.agentName, // or get from context
                        }
                    },
                },
                include: {
                    followUps: true,
                },
            });
            console.log("New lead created in agent lead repository:", newLead);
            // Notify Admin
            await this.notificationService.notifyAdmin({
                type: "lead_created",
                title: "New Agent Lead Created",
                message: `New lead created by agent ${leadData.agentName || 'Unknown Agent'}.`,
                entityType: "lead",
                entityId: newLead.id,
            });
            return this.formatLeadDates(newLead);
        }
        catch (error) {
            console.error("Error creating lead in agent lead repository:", error);
            throw error;
        }
    }
    //... all your other code remains the same
    async getAgentLeads(agentId) {
        try {
            const agent = await prismaClient_1.default.agent.findUnique({
                where: { id: agentId },
                select: { agentCode: true },
            });
            if (!agent || !agent.agentCode) {
                throw new Error("Agent not found or agentCode is missing. Agent ID:" + agentId);
            }
            const leads = await prismaClient_1.default.lead.findMany({
                where: {
                    agentCode: agent.agentCode,
                },
                include: {
                    followUps: true,
                },
            });
            return leads.map((lead) => this.formatLeadDates(lead));
        }
        catch (error) {
            console.error("Error getting agent leads in repository:", error);
            throw error;
        }
    }
    async getLeadById(leadId, agentId) {
        try {
            const agent = await prismaClient_1.default.agent.findUnique({
                where: { id: agentId },
                select: { agentCode: true },
            });
            if (!agent || !agent.agentCode) {
                throw new Error("Agent not found or agentCode is missing. Agent ID:" + agentId);
            }
            const lead = await prismaClient_1.default.lead.findUnique({
                where: {
                    id: leadId,
                    agentCode: agent.agentCode, // Agent can only see their leads
                },
                include: {
                    followUps: true,
                },
            });
            return lead ? this.formatLeadDates(lead) : null;
        }
        catch (error) {
            console.error("Error getting lead by ID in agent lead repository:", error);
            throw error;
        }
    }
    async updateLead(leadId, agentId, leadData) {
        try {
            const agent = await prismaClient_1.default.agent.findUnique({
                where: { id: agentId },
                select: { agentCode: true },
            });
            if (!agent || !agent.agentCode) {
                throw new Error("Agent not found or agentCode is missing. Agent ID:" + agentId);
            }
            const updatedLead = await prismaClient_1.default.lead.update({
                where: {
                    id: leadId,
                    agentCode: agent.agentCode, // Agent can only see their leads
                },
                data: leadData,
            });
            return updatedLead ? this.formatLeadDates(updatedLead) : null;
        }
        catch (error) {
            console.error("Error updating lead in agent lead repository:", error);
            return null;
        }
    }
    async deleteLead(leadId, agentId) {
        try {
            const agent = await prismaClient_1.default.agent.findUnique({
                where: { id: agentId },
                select: { agentCode: true },
            });
            if (!agent || !agent.agentCode) {
                throw new Error("Agent not found or agentCode is missing. Agent ID:" + agentId);
            }
            await prismaClient_1.default.lead.delete({
                where: {
                    id: leadId,
                    agentCode: agent.agentCode, // Agent can only see their leads
                },
            });
        }
        catch (error) {
            console.error("Error deleting lead in agent lead repository:", error);
            throw error;
        }
    }
    async updateLeadStatus(id, status) {
        try {
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: { status },
            });
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead status:", error);
            return null;
        }
    }
    async updateLeadStage(id, newStage, userName, notes) {
        try {
            // Get existing lead to fetch oldStage
            const existingLead = await prismaClient_1.default.lead.findUnique({
                where: { id },
                select: { stage: true },
            });
            if (!existingLead)
                return null;
            // Optional: Validate userId exists in DB to avoid FK violation
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: {
                    stage: newStage,
                    followUps: {
                        create: {
                            oldStage: existingLead.stage,
                            newStage: newStage,
                            userName: userName,
                            notes: notes || '',
                        },
                    },
                },
                include: { followUps: true },
            });
            console.log(updatedLead);
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead stage:", error);
            return null;
        }
    }
}
exports.AgentLeadRepository = AgentLeadRepository;
//# sourceMappingURL=AgentLeadRepository.js.map