"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TransferRepository = void 0;
const nanoid_1 = require("nanoid");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
// Enum to validate vehicle types
const validVehicles = [
    "MIDNIGHT_SURCHARGE",
    "ALPHARD_VELLFIRE",
    "EIGHTEEN_SEATER",
    "STAREX",
    "TEN_SEATER",
    "SEDAN",
];
class TransferRepository {
    async createTransfer(data) {
        // Validate vehicle type before inserting
        for (const vehicle of data.vehicleList) {
            if (!validVehicles.includes(vehicle.vehicle)) {
                throw new Error(`Invalid vehicle type: ${vehicle.vehicle}`);
            }
        }
        return await prismaClient_1.default.transfer.create({
            data: {
                id: (0, nanoid_1.nanoid)(24),
                name: data.name,
                from: data.from,
                to: data.to,
                supplierId: data.supplierId, // Corrected supplier reference
                terms: data.terms,
                vehicleList: {
                    create: data.vehicleList.map((vehicle) => ({
                        id: (0, nanoid_1.nanoid)(24),
                        vehicle: vehicle.vehicle,
                        pax: vehicle.pax,
                        price: vehicle.price,
                        optionalPrice: vehicle.optionalPrice,
                        minPax: vehicle.minPax,
                        maxPax: vehicle.maxPax,
                        transferId: data.id, // Corrected reference to match Transfer ID
                    })),
                },
            },
            include: { vehicleList: true }, // Corrected field name
        });
    }
    async getTransfers() {
        return await prismaClient_1.default.transfer.findMany({ include: { vehicleList: true, supplier: true } });
    }
    async getTransferById(id) {
        return await prismaClient_1.default.transfer.findUnique({
            where: { id },
            include: { vehicleList: true, supplier: true },
        });
    }
    async updateTransfer(id, data) {
        // Validate vehicle type before updating
        for (const vehicle of data.vehicleList) {
            if (!validVehicles.includes(vehicle.vehicle)) {
                throw new Error(`Invalid vehicle type: ${vehicle.vehicle}`);
            }
        }
        console.log("Data.vehicleList:", data.vehicleList);
        return await prismaClient_1.default.transfer.update({
            where: { id },
            data: {
                name: data.name,
                from: data.from,
                to: data.to,
                supplierId: data.supplierId,
                terms: data.terms,
                vehicleList: {
                    upsert: data.vehicleList.map((vehicle) => {
                        var _a;
                        return ({
                            where: { id: (_a = vehicle.id) !== null && _a !== void 0 ? _a : (0, nanoid_1.nanoid)(24) }, // Use existing ID or generate a new one
                            create: {
                                id: (0, nanoid_1.nanoid)(24),
                                vehicle: vehicle.vehicle,
                                pax: vehicle.pax,
                                price: vehicle.price,
                                optionalPrice: vehicle.optionalPrice,
                                minPax: vehicle.minPax,
                                maxPax: vehicle.maxPax,
                            },
                            update: {
                                vehicle: vehicle.vehicle,
                                pax: vehicle.pax,
                                price: vehicle.price,
                                optionalPrice: vehicle.optionalPrice,
                                minPax: vehicle.minPax,
                                maxPax: vehicle.maxPax,
                            },
                        });
                    }),
                },
            },
            include: { vehicleList: true }, // Corrected field name
        });
    }
    async deleteTransfer(id) {
        return await prismaClient_1.default.transfer.delete({
            where: { id },
            include: { vehicleList: true },
        });
    }
}
exports.TransferRepository = TransferRepository;
//# sourceMappingURL=transferRepository.js.map