"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TourRepository = void 0;
const client_1 = require("@prisma/client");
const nanoid_1 = require("nanoid");
const prisma = new client_1.PrismaClient();
class TourRepository {
    async createTour(data) {
        return await prisma.tour.create({
            data: {
                id: (0, nanoid_1.nanoid)(24),
                name: data.name,
                location: data.location,
                description: data.description,
                termsAndConditions: data.termsAndConditions,
                supplierId: data.supplierId,
                TourPrice: {
                    create: data.TourPrice.map((price) => ({
                        id: (0, nanoid_1.nanoid)(24),
                        minNoOfPax: price.minNoOfPax,
                        maxNoOfPax: price.maxNoOfPax,
                        paxType: price.paxType,
                        costPrice: price.costPrice,
                        optionalPrice: price.optionalPrice !== undefined && price.optionalPrice !== null
                            ? String(price.optionalPrice)
                            : null,
                        vehicle: price.vehicle
                    })),
                },
            },
            include: {
                TourPrice: true,
            },
        });
    }
    async getTours() {
        return await prisma.tour.findMany({
            include: {
                Suppliers: true,
                TourPrice: true,
            },
        });
    }
    async getTourById(id) {
        return await prisma.tour.findUnique({
            where: { id },
            include: {
                Suppliers: true,
                TourPrice: true,
            },
        });
    }
    async updateTour(id, data) {
        console.log("updateTour called with id:", id, "and data:", data);
        return await prisma.$transaction(async (prisma) => {
            const tourPricesToDelete = data.TourPrice.filter((price) => price._destroy && price.id);
            console.log("tourPricesToDelete:", tourPricesToDelete);
            // Delete tour prices using deleteMany
            if (tourPricesToDelete.length > 0) {
                try {
                    await prisma.tourPrice.deleteMany({
                        where: {
                            id: {
                                in: tourPricesToDelete.map((price) => price.id), // Ensure IDs are provided
                            },
                        },
                    });
                    console.log(`Successfully deleted tourPrices with IDs: ${tourPricesToDelete.map(price => price.id).join(', ')}`);
                }
                catch (error) {
                    console.error("Error deleting tour prices:", error);
                    throw error; // Re-throw to rollback the transaction
                }
            }
            const tourPricesToUpdate = data.TourPrice.filter(price => !price._destroy);
            console.log("tourPricesToUpdate:", tourPricesToUpdate);
            // Update the tour itself
            const updatedTour = await prisma.tour.update({
                where: { id },
                data: {
                    name: data.name,
                    location: data.location,
                    description: data.description,
                    termsAndConditions: data.termsAndConditions,
                    supplierId: data.supplierId,
                },
                include: {
                    TourPrice: true,
                },
            });
            // Now handle creating and updating tour prices separately
            for (const price of tourPricesToUpdate) {
                if (price.id) {
                    // Update existing price
                    try {
                        await prisma.tourPrice.update({
                            where: { id: price.id },
                            data: {
                                minNoOfPax: price.minNoOfPax,
                                maxNoOfPax: price.maxNoOfPax,
                                paxType: price.paxType,
                                costPrice: price.costPrice,
                                optionalPrice: price.optionalPrice,
                                vehicle: price.vehicle,
                            },
                        });
                        console.log(`Successfully updated tourPrice with ID: ${price.id}`);
                    }
                    catch (error) {
                        console.error(`Error updating tourPrice with ID: ${price.id}:`, error);
                        throw error; // Re-throw to rollback the transaction
                    }
                }
                else {
                    // Create new price
                    try {
                        await prisma.tourPrice.create({
                            data: {
                                id: (0, nanoid_1.nanoid)(24),
                                tourId: id, // Associate with the tour
                                minNoOfPax: price.minNoOfPax,
                                maxNoOfPax: price.maxNoOfPax,
                                paxType: price.paxType,
                                costPrice: price.costPrice,
                                optionalPrice: price.optionalPrice,
                                vehicle: price.vehicle,
                            },
                        });
                        console.log("Successfully created a new tourPrice");
                    }
                    catch (error) {
                        console.error("Error creating a new tourPrice:", error);
                        throw error; // Re-throw to rollback the transaction
                    }
                }
            }
            console.log("updatedTour:", updatedTour);
            return await prisma.tour.findUnique({
                where: { id },
                include: {
                    Suppliers: true,
                    TourPrice: true,
                },
            });
        });
    }
    async deleteTour(id) {
        return await prisma.tour.delete({
            where: { id },
            include: {
                TourPrice: true,
            },
        });
    }
}
exports.TourRepository = TourRepository;
//# sourceMappingURL=tourRepository.js.map