"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RestaurantRepository = void 0;
const cuid_1 = __importDefault(require("cuid"));
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class RestaurantRepository {
    async createRestaurant(data) {
        try {
            const restaurant = await prismaClient_1.default.restaurant.create({
                data: {
                    id: (0, cuid_1.default)(),
                    name: data.name,
                    location: data.location,
                    targetMarket: data.targetMarket,
                    description: data.description,
                    termsAndConditions: data.termsAndConditions,
                    supplierId: data.supplierId,
                    mealPrices: {
                        create: data.mealPrices.map((mp) => ({
                            category: mp.category,
                            pax: mp.pax,
                            costPrice: mp.costPrice,
                            optionalPrice: mp.optionalPrice,
                        })),
                    },
                },
                include: {
                    mealPrices: true,
                },
            });
            return restaurant;
        }
        catch (error) {
            console.error("Error creating restaurant:", error);
            throw new Error("Failed to create restaurant");
        }
    }
    async getRestaurants() {
        try {
            return await prismaClient_1.default.restaurant.findMany({
                include: {
                    mealPrices: true,
                    supplier: true
                },
            });
        }
        catch (error) {
            console.error("Error getting restaurants:", error);
            throw new Error("Failed to get restaurants");
        }
    }
    async getRestaurantById(id) {
        try {
            return await prismaClient_1.default.restaurant.findUnique({
                where: { id },
                include: {
                    mealPrices: true,
                    supplier: true
                },
            });
        }
        catch (error) {
            console.error(`Error getting restaurant with ID ${id}:`, error);
            throw new Error("Failed to get restaurant");
        }
    }
    async updateRestaurant(id, data) {
        try {
            // First, delete existing meal prices
            await prismaClient_1.default.mealPrice.deleteMany({
                where: { restaurantId: id },
            });
            // Then, update the restaurant and create new meal prices
            const restaurant = await prismaClient_1.default.restaurant.update({
                where: { id: id },
                data: {
                    name: data.name,
                    location: data.location,
                    targetMarket: data.targetMarket,
                    description: data.description,
                    termsAndConditions: data.termsAndConditions,
                    supplierId: data.supplierId,
                    mealPrices: {
                        create: data.mealPrices.map((mp) => ({
                            category: mp.category,
                            pax: mp.pax,
                            costPrice: mp.costPrice,
                            optionalPrice: mp.optionalPrice,
                        })),
                    },
                },
                include: {
                    mealPrices: true,
                }, // Moved include property outside of data
            });
            return restaurant;
        }
        catch (error) {
            console.error(`Error updating restaurant with ID ${id}:`, error);
            throw new Error("Failed to update restaurant");
        }
    }
    async deleteRestaurant(id) {
        try {
            return await prismaClient_1.default.restaurant.delete({
                where: { id },
            });
        }
        catch (error) {
            console.error(`Error deleting restaurant with ID ${id}:`, error);
            throw new Error("Failed to delete restaurant");
        }
    }
}
exports.RestaurantRepository = RestaurantRepository;
//# sourceMappingURL=restaurantrepository.js.map