"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuotationRepository = void 0;
const client_1 = require("@prisma/client");
const prisma = new client_1.PrismaClient();
class QuotationRepository {
    // Existing getDuplicateRefNo method (no changes needed here)
    getDuplicateRefNo(originalRefNo, existingRefNos) {
        // Remove any trailing -D{number} from the original refNo
        const baseRefNo = originalRefNo.replace(/-D\d+$/, "");
        // Find all duplicates of this base refNo
        const duplicates = existingRefNos.filter((ref) => ref.startsWith(`${baseRefNo}-D`));
        // Find highest -D{number}
        const maxDuplicateNumber = duplicates.reduce((max, ref) => {
            const match = ref.match(/-D(\d+)$/);
            const num = match ? parseInt(match[1], 10) : 0;
            return Math.max(max, num);
        }, 0);
        // Return next duplicate refNo
        return `${baseRefNo}-D${maxDuplicateNumber + 1}`;
    }
    async createQuotation(data, userName) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        try {
            const parseDate = (d) => d ? new Date(d) : null;
            const generateRefNo = async () => {
                var _a, _b;
                const leadId = (_b = (_a = data.leadRefNo) === null || _a === void 0 ? void 0 : _a.replace(/^LD-/, "")) !== null && _b !== void 0 ? _b : "UNKNOWN";
                const baseRef = `QT-${leadId}`;
                // Get only original and revised quotations (exclude duplicates like -D1, -D2)
                // This logic for `generateRefNo` is for revisions/new, not for -D duplicates.
                // It correctly finds the revision count based on QT-LEADID or QT-LEADID-R#
                const existingQuotations = await prisma.quotation.findMany({
                    where: {
                        leadRefNo: data.leadRefNo,
                        NOT: {
                            refNo: {
                                // Exclude any refNos ending with -D followed by digits
                                endsWith: '-D1', // This condition is too specific, needs to be more general
                                // A better approach would be to filter after fetching if performance is not an issue
                                // or use regex in Prisma if supported (it is for SQLite/Postgres via `raw`)
                                // For simplicity, we'll assume `startsWith` and filter in JS if not using raw
                            },
                        },
                    },
                    select: {
                        refNo: true,
                    },
                });
                const revisionCount = existingQuotations.filter(q => q.refNo === baseRef || (q.refNo.startsWith(`${baseRef}-R`) && !q.refNo.includes('-D'))).length;
                return revisionCount === 0 ? baseRef : `${baseRef}-R${revisionCount + 1}`;
            };
            let refNo;
            if ((_a = data.refNo) === null || _a === void 0 ? void 0 : _a.includes("-D")) {
                refNo = data.refNo; // This path won't be used by the duplicate function we are creating
            }
            else if (data.refNo) {
                // This implies 'data.refNo' is provided but doesn't have '-D', so it's a revision request
                refNo = await generateRefNo(); // Revision request
                console.log("Generated revision refNo:", refNo);
            }
            else {
                refNo = await generateRefNo(); // Fresh quotation (no refNo provided)
                console.log("Generated new refNo:", refNo);
            }
            const newStage = "quotation_created";
            const quotation = await prisma.quotation.create({
                data: {
                    refNo,
                    leadRefNo: data.leadRefNo,
                    leadAgent: (_b = data.leadAgent) !== null && _b !== void 0 ? _b : null,
                    status: (_c = data.status) !== null && _c !== void 0 ? _c : null,
                    guestName: (_d = data.guestName) !== null && _d !== void 0 ? _d : null,
                    remarks: (_e = data.remarks) !== null && _e !== void 0 ? _e : null,
                    termsAndConditions: (_f = data.termsAndConditions) !== null && _f !== void 0 ? _f : null,
                    importantNotes: (_g = data.importantNotes) !== null && _g !== void 0 ? _g : null,
                    currency: (_h = data.currency) !== null && _h !== void 0 ? _h : null,
                    currencyRate: (_j = data.currencyRate) !== null && _j !== void 0 ? _j : null,
                    totalBeforeMarkup: (_k = data.totalBeforeMarkup) !== null && _k !== void 0 ? _k : null,
                    totalAfterMarkup: (_l = data.totalAfterMarkup) !== null && _l !== void 0 ? _l : null,
                    travelDates: data.travelDates,
                    userid: (_m = data.userid) !== null && _m !== void 0 ? _m : null,
                    agentId: (_o = data.agentId) !== null && _o !== void 0 ? _o : null,
                    accommodations: {
                        create: ((_p = data.accommodations) !== null && _p !== void 0 ? _p : []).map((acc) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
                            return ({
                                hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                                hotelAddress: (_b = acc.hotelAddress) !== null && _b !== void 0 ? _b : null,
                                accMarkup: (_c = acc.accMarkup) !== null && _c !== void 0 ? _c : null,
                                supplierId: (_d = acc.supplierId) !== null && _d !== void 0 ? _d : null,
                                supplierName: (_e = acc.supplierName) !== null && _e !== void 0 ? _e : null,
                                supplierContact: (_f = acc.supplierContact) !== null && _f !== void 0 ? _f : null,
                                room: (_g = acc.room) !== null && _g !== void 0 ? _g : null,
                                travelDate: parseDate(acc.travelDate),
                                endDate: parseDate(acc.endDate),
                                nights: (_h = acc.nights) !== null && _h !== void 0 ? _h : null,
                                surchargeNights: (_j = acc.surchargeNights) !== null && _j !== void 0 ? _j : null,
                                quantity: (_k = acc.quantity) !== null && _k !== void 0 ? _k : null,
                                totalPrice: (_l = acc.totalPrice) !== null && _l !== void 0 ? _l : null,
                                totalPriceAfterMarkup: (_m = acc.totalPriceAfterMarkup) !== null && _m !== void 0 ? _m : null,
                                priceDetailsAccommodation: {
                                    create: ((_o = acc.priceDetails) !== null && _o !== void 0 ? _o : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                                        return ({
                                            roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                            category: (_b = pd.category) !== null && _b !== void 0 ? _b : null,
                                            pax: (_c = pd.pax) !== null && _c !== void 0 ? _c : null,
                                            price: (_d = pd.price) !== null && _d !== void 0 ? _d : null,
                                            priceAfterMarkup: (_e = pd.priceAfterMarkup) !== null && _e !== void 0 ? _e : null,
                                            costPrice: (_f = pd.costPrice) !== null && _f !== void 0 ? _f : null,
                                            optionalPrice: (_g = pd.optionalPrice) !== null && _g !== void 0 ? _g : null,
                                            quantity: (_h = pd.quantity) !== null && _h !== void 0 ? _h : null,
                                            totalPrice: (_j = pd.totalPrice) !== null && _j !== void 0 ? _j : null,
                                            totalPriceAfterMarkup: (_k = pd.totalPriceAfterMarkup) !== null && _k !== void 0 ? _k : null,
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                    itineraries: {
                        create: ((_q = data.itineraries) !== null && _q !== void 0 ? _q : []).map((itin) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                            return ({
                                date: parseDate(itin.date),
                                label: (_a = itin.label) !== null && _a !== void 0 ? _a : null,
                                category: (_b = itin.category) !== null && _b !== void 0 ? _b : null,
                                itinMarkup: (_c = itin.itinMarkup) !== null && _c !== void 0 ? _c : null,
                                supplierId: (_d = itin.supplierId) !== null && _d !== void 0 ? _d : null,
                                supplierName: (_e = itin.supplierName) !== null && _e !== void 0 ? _e : null,
                                supplierContact: (_f = itin.supplierContact) !== null && _f !== void 0 ? _f : null,
                                totalPriceAfterMarkup: (_g = itin.totalPriceAfterMarkup) !== null && _g !== void 0 ? _g : null,
                                priceDetailsList: {
                                    create: ((_h = itin.priceDetails) !== null && _h !== void 0 ? _h : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f;
                                        return ({
                                            category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                            price: (_b = pd.price) !== null && _b !== void 0 ? _b : null,
                                            priceAfterMarkup: (_c = pd.priceAfterMarkup) !== null && _c !== void 0 ? _c : null,
                                            quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                            totalPrice: (_e = pd.totalPrice) !== null && _e !== void 0 ? _e : null,
                                            totalPriceAfterMarkup: (_f = pd.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                        });
                                    }),
                                },
                                guideId: (_j = itin.guideId) !== null && _j !== void 0 ? _j : null,
                            });
                        }),
                    },
                    tours: {
                        connect: ((_r = data.tours) !== null && _r !== void 0 ? _r : []).map((tourId) => ({
                            id: tourId,
                        })),
                    },
                },
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            // Update Lead stage and create follow-up
            const lead = await prisma.lead.findUnique({
                where: { leadNo: data.leadRefNo },
            });
            if (lead) {
                await prisma.lead.update({
                    where: { id: lead.id },
                    data: { stage: newStage },
                });
                await prisma.followUp.create({
                    data: {
                        leadId: lead.id,
                        oldStage: lead.stage,
                        newStage: newStage,
                        notes: "Quotation created",
                        userName: userName !== null && userName !== void 0 ? userName : null,
                    },
                });
            }
            console.log("Quotation created in repo: ******************", quotation.refNo);
            return quotation;
        }
        catch (error) {
            console.error("Error creating quotation:", error.message);
            throw new Error("Failed to create quotation");
        }
    }
    // --- NEW DUPLICATION METHOD ---
    async duplicateQuotation(id, userName) {
        var _a, _b;
        try {
            const originalQuotation = await prisma.quotation.findUnique({
                where: { id },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true, // Make sure to include tours for copying
                },
            });
            if (!originalQuotation) {
                throw new Error("Original quotation not found for duplication");
            }
            // 1. Get all refNos for this lead to determine the next duplicate number
            const allQuotationsForLead = await prisma.quotation.findMany({
                where: { leadRefNo: originalQuotation.leadRefNo },
                select: { refNo: true },
            });
            const existingRefNosForLead = allQuotationsForLead.map((q) => q.refNo);
            // The base for duplication should be the original refNo without any -D suffix,
            // as new duplicates will be appended to this base.
            const baseRefForDuplication = originalQuotation.refNo.replace(/-D\d+$/, "");
            const newRefNo = this.getDuplicateRefNo(baseRefForDuplication, existingRefNosForLead);
            // 2. Prepare data for the new quotation
            // Dates from Prisma are already Date objects, so parseDate needs adjustment for this context
            const parseDate = (d) => (d ? d : null);
            const duplicateData = {
                refNo: newRefNo,
                leadRefNo: originalQuotation.leadRefNo,
                leadAgent: originalQuotation.leadAgent,
                status: client_1.QuotationStatus.Draft, // Set duplicated quotation status to 'Draft'
                guestName: originalQuotation.guestName,
                remarks: `Duplicate of ${originalQuotation.refNo} (created by ${userName || 'System'})`, // Add a remark
                termsAndConditions: (_a = originalQuotation.termsAndConditions) !== null && _a !== void 0 ? _a : null,
                importantNotes: (_b = originalQuotation.importantNotes) !== null && _b !== void 0 ? _b : null,
                currency: originalQuotation.currency,
                currencyRate: originalQuotation.currencyRate,
                totalBeforeMarkup: originalQuotation.totalBeforeMarkup,
                totalAfterMarkup: originalQuotation.totalAfterMarkup,
                travelDates: originalQuotation.travelDates,
                userid: originalQuotation.userid,
                agentId: originalQuotation.agentId,
                // createdAt and updatedAt will be automatically set by Prisma for the new record
            };
            // Handle nested relations for creation
            duplicateData.accommodations = {
                create: originalQuotation.accommodations.map((acc) => {
                    var _a;
                    return ({
                        hotel: acc.hotel,
                        hotelAddress: acc.hotelAddress,
                        accMarkup: (_a = acc.accMarkup) !== null && _a !== void 0 ? _a : null,
                        supplierId: acc.supplierId,
                        supplierName: acc.supplierName,
                        supplierContact: acc.supplierContact,
                        room: acc.room,
                        travelDate: parseDate(acc.travelDate),
                        endDate: parseDate(acc.endDate),
                        nights: acc.nights,
                        surchargeNights: acc.surchargeNights,
                        quantity: acc.quantity,
                        totalPrice: acc.totalPrice,
                        totalPriceAfterMarkup: acc.totalPriceAfterMarkup,
                        priceDetailsAccommodation: {
                            create: acc.priceDetailsAccommodation.map((pd) => ({
                                roomId: pd.roomId,
                                category: pd.category,
                                pax: pd.pax,
                                price: pd.price,
                                priceAfterMarkup: pd.priceAfterMarkup,
                                costPrice: pd.costPrice,
                                optionalPrice: pd.optionalPrice,
                                quantity: pd.quantity,
                                totalPrice: pd.totalPrice,
                                totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                            })),
                        },
                    });
                }),
            };
            duplicateData.itineraries = {
                create: originalQuotation.itineraries.map((itin) => {
                    var _a;
                    return ({
                        date: parseDate(itin.date),
                        label: itin.label,
                        category: itin.category,
                        itinMarkup: (_a = itin.itinMarkup) !== null && _a !== void 0 ? _a : null,
                        supplierId: itin.supplierId,
                        supplierName: itin.supplierName,
                        supplierContact: itin.supplierContact,
                        totalPriceAfterMarkup: itin.totalPriceAfterMarkup,
                        priceDetailsList: {
                            create: itin.priceDetailsList.map((pd) => ({
                                category: pd.category,
                                price: pd.price,
                                priceAfterMarkup: pd.priceAfterMarkup,
                                quantity: pd.quantity,
                                totalPrice: pd.totalPrice,
                                totalPriceAfterMarkup: pd.totalPriceAfterMarkup,
                            })),
                        },
                        guideId: itin.guideId,
                    });
                }),
            };
            // For many-to-many relationships (like tours), use `connect` to link existing tours
            duplicateData.tours = {
                connect: originalQuotation.tours.map((tour) => ({ id: tour.id })),
            };
            const duplicatedQuotation = await prisma.quotation.create({
                data: duplicateData,
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            console.log("Quotation duplicated in repo:", duplicatedQuotation.refNo);
            return duplicatedQuotation;
        }
        catch (error) {
            console.error("Error duplicating quotation:", error.message);
            throw new Error("Failed to duplicate quotation");
        }
    }
    async getQuotationByLeadRefNo(leadRefNo) {
        try {
            const quotation = await prisma.quotation.findMany({
                where: { leadRefNo },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true }
                    },
                    tours: true,
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                            User: true, // Include user details
                        },
                    },
                },
            });
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            console.log("Quotation in repo", quotation);
            return quotation;
        }
        catch (error) {
            console.error("Error fetching quotation:", error);
            throw error; // Re-throw to handle in the calling function
        }
    }
    async getQuotationById(id) {
        try {
            const quotation = await prisma.quotation.findUnique({
                where: { id },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true,
                    user: true, // Include user details  
                    lead: {
                        include: {
                            Agent: true, // Note: Capital 'A' to match your schema
                            User: true, // Include user details
                        },
                    },
                },
            });
            if (!quotation) {
                throw new Error("Quotation not found");
            }
            console.log("Quotation in repo", quotation);
            return quotation;
        }
        catch (error) {
            console.error("Error fetching quotation:", error);
            throw error; // Re-throw to handle in the calling function
        }
    }
    async getAllQuotations() {
        try {
            const quotations = await prisma.quotation.findMany({
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true }, // Fix incorrect reference
                    },
                    tours: true,
                },
            });
            return quotations;
        }
        catch (error) {
            console.error("Error fetching quotations:", error);
            throw new Error("Could not fetch quotations");
        }
    }
    async updateQuotation(id, data) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        try {
            const parseDate = (d) => d ? new Date(d) : null;
            // Delete nested children first
            await prisma.priceDetailsAccommodation.deleteMany({
                where: { accommodation: { quotationId: id } },
            });
            await prisma.accommodation.deleteMany({ where: { quotationId: id } });
            await prisma.priceDetailItinerary.deleteMany({
                where: { itinerary: { quotationId: id } },
            });
            await prisma.itinerary.deleteMany({ where: { quotationId: id } });
            await prisma.quotation.update({
                where: { id },
                data: {
                    tours: {
                        set: [],
                    },
                },
            });
            const updatedQuotation = await prisma.quotation.update({
                where: { id },
                data: {
                    leadRefNo: data.leadRefNo,
                    leadAgent: (_a = data.leadAgent) !== null && _a !== void 0 ? _a : null,
                    status: (_b = data.status) !== null && _b !== void 0 ? _b : null,
                    guestName: (_c = data.guestName) !== null && _c !== void 0 ? _c : null,
                    remarks: (_d = data.remarks) !== null && _d !== void 0 ? _d : null,
                    termsAndConditions: (_e = data.termsAndConditions) !== null && _e !== void 0 ? _e : null,
                    importantNotes: (_f = data.importantNotes) !== null && _f !== void 0 ? _f : null,
                    currency: (_g = data.currency) !== null && _g !== void 0 ? _g : null,
                    currencyRate: (_h = data.currencyRate) !== null && _h !== void 0 ? _h : null,
                    totalBeforeMarkup: (_j = data.totalBeforeMarkup) !== null && _j !== void 0 ? _j : null,
                    totalAfterMarkup: (_k = data.totalAfterMarkup) !== null && _k !== void 0 ? _k : null,
                    travelDates: data.travelDates,
                    userid: (_l = data.userid) !== null && _l !== void 0 ? _l : null,
                    agentId: (_m = data.agentId) !== null && _m !== void 0 ? _m : null,
                    accommodations: {
                        create: ((_o = data.accommodations) !== null && _o !== void 0 ? _o : []).map((acc) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
                            return ({
                                hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                                hotelAddress: (_b = acc.hotelAddress) !== null && _b !== void 0 ? _b : null,
                                accMarkup: (_c = acc.accMarkup) !== null && _c !== void 0 ? _c : null,
                                supplierId: (_d = acc.supplierId) !== null && _d !== void 0 ? _d : null,
                                supplierName: (_e = acc.supplierName) !== null && _e !== void 0 ? _e : null,
                                supplierContact: (_f = acc.supplierContact) !== null && _f !== void 0 ? _f : null,
                                room: (_g = acc.room) !== null && _g !== void 0 ? _g : null,
                                travelDate: parseDate(acc.travelDate),
                                endDate: parseDate(acc.endDate),
                                nights: (_h = acc.nights) !== null && _h !== void 0 ? _h : null,
                                surchargeNights: (_j = acc.surchargeNights) !== null && _j !== void 0 ? _j : null,
                                quantity: (_k = acc.quantity) !== null && _k !== void 0 ? _k : null,
                                totalPrice: (_l = acc.totalPrice) !== null && _l !== void 0 ? _l : null,
                                totalPriceAfterMarkup: (_m = acc.totalPriceAfterMarkup) !== null && _m !== void 0 ? _m : null,
                                priceDetailsAccommodation: {
                                    create: ((_o = acc.priceDetails) !== null && _o !== void 0 ? _o : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                                        return ({
                                            roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                            category: (_b = pd.category) !== null && _b !== void 0 ? _b : null,
                                            pax: (_c = pd.pax) !== null && _c !== void 0 ? _c : null,
                                            price: (_d = pd.price) !== null && _d !== void 0 ? _d : null,
                                            priceAfterMarkup: (_e = pd.priceAfterMarkup) !== null && _e !== void 0 ? _e : null,
                                            costPrice: (_f = pd.costPrice) !== null && _f !== void 0 ? _f : null,
                                            optionalPrice: (_g = pd.optionalPrice) !== null && _g !== void 0 ? _g : null,
                                            quantity: (_h = pd.quantity) !== null && _h !== void 0 ? _h : null,
                                            totalPrice: (_j = pd.totalPrice) !== null && _j !== void 0 ? _j : null,
                                            totalPriceAfterMarkup: (_k = pd.totalPriceAfterMarkup) !== null && _k !== void 0 ? _k : null,
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                    itineraries: {
                        create: ((_p = data.itineraries) !== null && _p !== void 0 ? _p : []).map((itin) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                            return ({
                                date: parseDate(itin.date),
                                label: (_a = itin.label) !== null && _a !== void 0 ? _a : null,
                                category: (_b = itin.category) !== null && _b !== void 0 ? _b : null,
                                itinMarkup: (_c = itin.itinMarkup) !== null && _c !== void 0 ? _c : null,
                                supplierId: (_d = itin.supplierId) !== null && _d !== void 0 ? _d : null,
                                supplierName: (_e = itin.supplierName) !== null && _e !== void 0 ? _e : null,
                                supplierContact: (_f = itin.supplierContact) !== null && _f !== void 0 ? _f : null,
                                totalPriceAfterMarkup: (_g = itin.totalPriceAfterMarkup) !== null && _g !== void 0 ? _g : null,
                                priceDetailsList: {
                                    create: ((_h = itin.priceDetails) !== null && _h !== void 0 ? _h : []).map((pd) => {
                                        var _a, _b, _c, _d, _e, _f;
                                        return ({
                                            category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                            price: (_b = pd.price) !== null && _b !== void 0 ? _b : null,
                                            priceAfterMarkup: (_c = pd.priceAfterMarkup) !== null && _c !== void 0 ? _c : null,
                                            quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                            totalPrice: (_e = pd.totalPrice) !== null && _e !== void 0 ? _e : null,
                                            totalPriceAfterMarkup: (_f = pd.totalPriceAfterMarkup) !== null && _f !== void 0 ? _f : null,
                                        });
                                    }),
                                },
                                guideId: (_j = itin.guideId) !== null && _j !== void 0 ? _j : null, // ADDED guideId
                            });
                        }),
                    },
                    tours: {
                        connect: ((_q = data.tours) !== null && _q !== void 0 ? _q : []).map((tourId) => ({
                            id: tourId,
                        })),
                    },
                },
                include: {
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                    tours: true,
                },
            });
            console.log("Updtaed Quotation", updatedQuotation);
            return updatedQuotation;
        }
        catch (error) {
            console.error("Error updating quotation:", error.message);
            throw new Error("Failed to update quotation");
        }
    }
    async deleteQuotation(id) {
        console.log("ID in repository:", id); // Debugging
        const existingQuotation = await prisma.quotation.findUnique({
            where: { id },
        });
        if (!existingQuotation) {
            throw new Error("Quotation not found");
        }
        return await prisma.quotation.delete({
            where: { id },
        });
    }
    async updateQuotationStatus(id, status, userName) {
        try {
            const updatedQuotation = await prisma.quotation.update({
                where: { id },
                data: { status },
                include: {
                    lead: true, // so we can access lead data
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    tours: true,
                },
            });
            if (!updatedQuotation)
                throw new Error("Quotation not found");
            // If status is 'Accepted', also update Lead and create FollowUp
            if (status === 'Accepted') {
                const leadUpdate = await prisma.lead.update({
                    where: { leadNo: updatedQuotation.leadRefNo },
                    data: {
                        stage: 'quotation_accepted',
                        followUps: {
                            create: {
                                oldStage: updatedQuotation.lead.stage,
                                newStage: 'quotation_accepted',
                                notes: `Quotation ${updatedQuotation.refNo} accepted.`,
                                userName: userName,
                            },
                        },
                    },
                });
            }
            return updatedQuotation;
        }
        catch (error) {
            console.error("Error updating quotation status:", error);
            throw error;
        }
    }
    async getAcceptedQuotationsByLeadId(leadId) {
        try {
            const lead = await prisma.lead.findUnique({
                where: { id: leadId },
                include: {
                    Quotation: {
                        where: { status: 'Accepted' },
                        include: {
                            accommodations: true,
                            itineraries: true,
                            tours: true,
                        },
                    },
                },
            });
            if (!lead)
                throw new Error("Lead not found");
            return lead;
        }
        catch (error) {
            console.error("Error fetching accepted quotations:", error);
            throw error;
        }
    }
}
exports.QuotationRepository = QuotationRepository;
//# sourceMappingURL=quotationRepository.js.map