"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LeadRepository = void 0;
const date_fns_1 = require("date-fns");
const prismaClient_1 = __importDefault(require("../../config/prismaClient")); // Import Prisma client
const notificationService_1 = require("../../services/common/notificationService");
const UserbookingRepository_1 = require("../../repositories/user/UserbookingRepository");
class LeadRepository {
    constructor() {
        this.notificationService = new notificationService_1.NotificationService();
        this.bookingRepository = new UserbookingRepository_1.BookingRepository();
    }
    formatLeadDates(lead) {
        return {
            ...lead,
            travelDateFrom: lead.travelDateFrom
                ? new Date(lead.travelDateFrom).toISOString()
                : null,
            travelDateTo: lead.travelDateTo
                ? new Date(lead.travelDateTo).toISOString()
                : null,
        };
    }
    async createLead(leadData, userName) {
        var _a, _b;
        try {
            console.log("LeadRepository createLead input:", leadData);
            const newLead = await prismaClient_1.default.lead.create({
                data: {
                    ...leadData,
                    agentId: (_a = leadData.agentId) !== null && _a !== void 0 ? _a : null, // Ensure agentId is explicitly set to null if undefined
                    userId: (_b = leadData.userId) !== null && _b !== void 0 ? _b : null, // Ensure userId is explicitly set to null if undefined
                    followUps: {
                        create: {
                            oldStage: 'new_lead',
                            newStage: 'new_lead',
                            notes: 'Lead created',
                            userName: userName, // or get from context
                        }
                    }
                },
                include: {
                    followUps: true,
                }
            });
            return this.formatLeadDates(newLead);
        }
        catch (error) {
            console.error("Error creating lead in repository:", error);
            throw error;
        }
    }
    async getAllLeads() {
        try {
            const leadsWithRelations = await prismaClient_1.default.lead.findMany({
                include: {
                    Agent: true,
                    User: true,
                },
                orderBy: {
                    createdOn: 'desc',
                }
            });
            return leadsWithRelations.map(this.formatLeadDates);
        }
        catch (error) {
            console.error("Error getting leads in repository:", error);
            throw error;
        }
    }
    async getLeadById(leadId) {
        try {
            const lead = await prismaClient_1.default.lead.findUnique({
                where: { id: leadId },
                include: {
                    Agent: true, // Include agent details
                    User: true, // Include user details
                    followUps: true,
                    Quotation: true,
                },
            });
            return lead ? this.formatLeadDates(lead) : null;
        }
        catch (error) {
            console.error("Error getting lead by ID in repository:", error);
            throw error;
        }
    }
    async updateLead(leadId, leadData) {
        try {
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id: leadId },
                data: {
                    empCode: leadData.empCode,
                    agentName: leadData.agentName,
                    agentCode: leadData.agentCode,
                    leadNo: leadData.leadNo,
                    travelDateFrom: leadData.travelDateFrom,
                    travelDateTo: leadData.travelDateTo,
                    totalPax: leadData.totalPax,
                    description: leadData.description,
                    customerName: leadData.customerName,
                    email: leadData.email,
                    phone: leadData.phone,
                    adults: Number(leadData.adults), // ✅ Convert to number
                    kids: Number(leadData.kids), // ✅ Convert to number
                    infants: Number(leadData.infants), // ✅ Convert to number
                    remarks: leadData.remarks,
                    agentId: leadData.agentId,
                    userId: leadData.userId, // Ensure userId is included if provided
                },
            });
            console.log("Updated Lead:", updatedLead);
            return updatedLead ? this.formatLeadDates(updatedLead) : null;
        }
        catch (error) {
            console.error("Error updating lead in repository:", error);
            return null;
        }
    }
    async deleteLead(leadId) {
        try {
            await prismaClient_1.default.lead.delete({
                where: { id: leadId },
            });
        }
        catch (error) {
            console.error("Error deleting lead in repository:", error);
            throw error;
        }
    }
    async getLeadsByTimeframe(timeframe) {
        let startDate;
        let endDate = new Date(); // Default end date is now
        switch (timeframe) {
            case "today":
                startDate = (0, date_fns_1.startOfDay)(new Date());
                break;
            case "weekly":
                startDate = (0, date_fns_1.startOfWeek)(new Date(), { weekStartsOn: 1 }); // Monday
                break;
            case "month":
                startDate = (0, date_fns_1.startOfMonth)(new Date());
                break;
            case "year":
                startDate = (0, date_fns_1.startOfYear)(new Date());
                break;
            default:
                throw new Error(`Invalid timeframe: ${timeframe}`);
        }
        try {
            const leads = await prismaClient_1.default.lead.findMany({
                where: {
                    createdOn: {
                        gte: startDate,
                        lte: endDate,
                    },
                },
                select: {
                    createdOn: true,
                    stage: true,
                    status: true,
                },
            });
            const groupedLeads = this.formatLeadsByTimeframe(leads, timeframe);
            console.log("Grouped Leads:", groupedLeads);
            return groupedLeads;
        }
        catch (error) {
            console.error("Error fetching leads by timeframe:", error);
            throw error;
        }
    }
    formatLeadsByTimeframe(leads, timeframe) {
        const grouped = {};
        for (const lead of leads) {
            let groupKey;
            if (timeframe === "today" || timeframe === "weekly") {
                groupKey = (0, date_fns_1.format)(lead.createdOn, "EEEE"); // e.g., "Monday"
            }
            else if (timeframe === "month") {
                groupKey = `Week ${Math.ceil(lead.createdOn.getDate() / 7)}`; // e.g., "Week 1"
            }
            else if (timeframe === "year") {
                groupKey = (0, date_fns_1.format)(lead.createdOn, "MMM"); // e.g., "Jan"
            }
            else {
                groupKey = (0, date_fns_1.format)(lead.createdOn, "yyyy-MM-dd");
            }
            if (!grouped[groupKey]) {
                grouped[groupKey] = {
                    day: groupKey,
                    leads: 0,
                    booked: 0,
                    cancelled: 0,
                };
            }
            grouped[groupKey].leads++;
            if (lead.stage === "quotation_accepted" || lead.stage === "trip_started" || lead.stage === "trip_completed" || lead.stage === "assigned_to_operations" || lead.stage === "assigned_to_assessment") {
                grouped[groupKey].booked++;
            }
            if (lead.stage === "canceled") {
                grouped[groupKey].cancelled++;
            }
        }
        return Object.values(grouped);
    }
    async updateLeadStatus(id, status) {
        try {
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: { status },
            });
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead status:", error);
            return null;
        }
    }
    async updateLeadStage(id, newStage, userName, notes) {
        try {
            // Get existing lead to fetch oldStage
            const existingLead = await prismaClient_1.default.lead.findUnique({
                where: { id },
                select: { stage: true },
            });
            if (!existingLead)
                return null;
            // Optional: Validate userId exists in DB to avoid FK violation
            const updatedLead = await prismaClient_1.default.lead.update({
                where: { id },
                data: {
                    stage: newStage,
                    followUps: {
                        create: {
                            oldStage: existingLead.stage,
                            newStage: newStage,
                            userName: userName,
                            notes: notes || '',
                        },
                    },
                },
                include: { followUps: true },
            });
            console.log(updatedLead);
            return this.formatLeadDates(updatedLead);
        }
        catch (error) {
            console.error("Error updating lead stage:", error);
            return null;
        }
    }
    async assignLeadToOperations(leadId, operationUserId) {
        try {
            const updatedLead = await prismaClient_1.default.$transaction(async (tx) => {
                // 1. Get the current lead and its stage
                const lead = await tx.lead.findUnique({
                    where: { id: leadId },
                    select: { stage: true, Booking: true, leadNo: true }, // Select related Booking object
                });
                if (!lead) {
                    throw new Error("Lead not found.");
                }
                // 2. Get the operation user details
                const operationUser = await tx.user.findUnique({
                    where: { id: operationUserId },
                    select: { name: true },
                });
                if (!operationUser) {
                    throw new Error("Operation user not found.");
                }
                const operationUserName = operationUser.name;
                // 3. Update the lead with assignedOperationId and new stage
                const updated = await tx.lead.update({
                    where: { id: leadId },
                    data: {
                        assignedOperationId: operationUserId,
                        stage: 'assigned_to_operations',
                    },
                });
                // 4. Add a follow-up entry
                await tx.followUp.create({
                    data: {
                        id: crypto.randomUUID(),
                        date: new Date(),
                        oldStage: lead.stage,
                        newStage: 'assigned_to_operations',
                        notes: `Assigned to ${operationUserName}, Operation Team`,
                        userName: 'Admin',
                        leadId: leadId,
                    },
                });
                return updated;
            });
            // Find the booking ID related to the lead
            const booking = await prismaClient_1.default.booking.findFirst({
                where: {
                    lead: {
                        id: leadId,
                    },
                },
            });
            const bookingId = booking === null || booking === void 0 ? void 0 : booking.id;
            console.log(bookingId, "booking id from assigned to operation ");
            // Send notification to the operations user
            await this.notificationService.notifyUser(operationUserId, {
                type: "lead_assigned_operations",
                title: "Lead Assigned to Operations",
                message: `Lead ${updatedLead.leadNo} has been assigned to you in the Operations team.`, // Use leadNo if available
                entityType: "lead", // Use booking as entity type
                entityId: bookingId || leadId, // Use booking ID if exists, else use lead ID
            });
            return updatedLead;
        }
        catch (error) {
            console.error('Error assigning lead to operations user:', error);
            throw new Error('Failed to assign lead to operations user.');
        }
    }
    async assignLeadToAssessment(leadId, assessmentUserId) {
        try {
            // Input Validation: Important for security
            if (!leadId || !assessmentUserId) {
                throw new Error("Lead ID and Assessment User ID are required.");
            }
            const updatedLead = await prismaClient_1.default.$transaction(async (tx) => {
                // 1. Get the current lead and its stage, assignedToId, and assignedAssessmentId
                const lead = await tx.lead.findUnique({
                    where: { id: leadId },
                    select: {
                        stage: true,
                        assignedToId: true,
                        leadNo: true,
                        assignedAssessmentId: true,
                        assignedAssessmentOn: true,
                    },
                });
                if (!lead) {
                    throw new Error("Lead not found.");
                }
                // Check if the lead is already assigned to this assessment user
                if (lead.assignedAssessmentId === assessmentUserId) {
                    throw new Error("Lead is already assigned to this assessment user.");
                }
                // 2. Get the assessment user details
                const assessmentUser = await tx.user.findUnique({
                    where: { id: assessmentUserId },
                    select: { name: true },
                });
                if (!assessmentUser) {
                    throw new Error("Assessment user not found.");
                }
                const assessmentUserName = assessmentUser.name;
                // 3. Update the lead with assignedAssessmentId and new stage
                const updated = await tx.lead.update({
                    where: { id: leadId },
                    data: {
                        assignedAssessmentId: assessmentUserId, // Assign the lead to the assessment user
                        stage: 'assigned_to_assessment',
                        assignedAssessmentOn: new Date(),
                    },
                });
                // 4. Add a follow-up entry
                await tx.followUp.create({
                    data: {
                        id: crypto.randomUUID(),
                        date: new Date(),
                        oldStage: lead.stage,
                        newStage: 'assigned_to_assessment',
                        notes: `Assigned to ${assessmentUserName} for assessment.`,
                        userName: 'Admin', // Or the current user's name
                        leadId: leadId,
                    },
                });
                return updated;
            });
            // Find the booking ID related to the lead
            const booking = await prismaClient_1.default.booking.findFirst({
                where: {
                    lead: { id: leadId },
                },
            });
            const bookingId = booking === null || booking === void 0 ? void 0 : booking.id;
            await this.notificationService.notifyUser(assessmentUserId, {
                type: "lead_assigned_assessment",
                title: "Lead Assigned to Assessment",
                message: `Lead ${updatedLead.leadNo} has been assigned to you for assessment. Please review it.`,
                entityType: "assessment", // Consistent with Operations - using Lead entity type
                entityId: bookingId || leadId,
            });
            return updatedLead;
        }
        catch (error) {
            console.error('Error assigning lead to assessment:', error);
            throw new Error('Failed to assign lead to assessment.');
        }
    }
    async findQuotationAcceptedLeads() {
        try {
            console.log("repo called");
            const acceptedLead = await prismaClient_1.default.lead.findMany({
                where: {
                    stage: "quotation_accepted",
                },
                include: {
                    Agent: true,
                    Quotation: {
                        where: {
                            status: "Accepted",
                        },
                        include: {
                            itineraries: {
                                include: {
                                    priceDetailsList: true,
                                },
                            },
                            accommodations: {
                                include: {
                                    priceDetailsAccommodation: true,
                                },
                            },
                        },
                    },
                },
            });
            console.log(acceptedLead);
            return acceptedLead;
        }
        catch (error) {
            console.error("Error fetching quotation accepted leads:", error);
            throw new Error("Failed to fetch leads with stage 'quotation_accepted'");
        }
    }
}
exports.LeadRepository = LeadRepository;
//# sourceMappingURL=leadRepository.js.map