"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HotelRepository = void 0;
const client_1 = require("@prisma/client");
const cuid_1 = __importDefault(require("cuid"));
const prisma = new client_1.PrismaClient({
    log: ['query', 'info', 'warn', 'error'],
});
class HotelRepository {
    async createHotel(data) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        try {
            console.log("Data being passed to createHotel:", JSON.stringify(data, null, 2));
            const createdHotel = await prisma.hotels.create({
                data: {
                    id: (_a = data.id) !== null && _a !== void 0 ? _a : (0, cuid_1.default)(), // Use provided ID or generate a new one
                    name: (_b = data.name) !== null && _b !== void 0 ? _b : "",
                    location: (_c = data.location) !== null && _c !== void 0 ? _c : "",
                    starRating: (_d = data.starRating) !== null && _d !== void 0 ? _d : "",
                    tourismTax: (_e = data.tourismTax) !== null && _e !== void 0 ? _e : "",
                    stateTax: (_f = data.stateTax) !== null && _f !== void 0 ? _f : "",
                    address: (_g = data.address) !== null && _g !== void 0 ? _g : "",
                    phone: (_h = data.phone) !== null && _h !== void 0 ? _h : "",
                    description: (_j = data.description) !== null && _j !== void 0 ? _j : "",
                    termsAndConditions: (_k = data.termsAndConditions) !== null && _k !== void 0 ? _k : "",
                    supplierId: (_l = data.supplierId) !== null && _l !== void 0 ? _l : null,
                    surcharge: (_m = data.surcharge) !== null && _m !== void 0 ? _m : {},
                    closedDates: (_o = data.closedDates) !== null && _o !== void 0 ? _o : {},
                    price: data.price
                        ? {
                            create: data.price.map((p) => {
                                var _a, _b, _c, _d;
                                return ({
                                    category: (_a = p.category) !== null && _a !== void 0 ? _a : "",
                                    pax: (_b = p.pax) !== null && _b !== void 0 ? _b : "",
                                    costPrice: (_c = p.costPrice) !== null && _c !== void 0 ? _c : 0,
                                    optionalPrice: (_d = p.optionalPrice) !== null && _d !== void 0 ? _d : 0,
                                });
                            }),
                        }
                        : undefined,
                },
            });
            console.log("Hotel created successfully:", createdHotel);
            return createdHotel;
        }
        catch (error) {
            console.error("Error creating hotel:", error);
            throw new Error("Failed to create hotel");
        }
    }
    async getHotels(page = 1, limit = 10, searchTerm = "") {
        try {
            const skip = (page - 1) * limit;
            const hotels = await prisma.hotels.findMany({
                where: {
                    name: {
                        contains: searchTerm,
                        mode: "insensitive",
                    },
                },
                orderBy: {
                    createdAt: "desc",
                },
                skip,
                take: limit,
                include: {
                    rooms: true,
                    price: true,
                    supplier: {
                        select: {
                            id: true,
                            companyName: true, // Fetching only required supplier details
                        },
                    },
                },
            });
            const totalHotels = await prisma.hotels.count({
                where: {
                    name: {
                        contains: searchTerm,
                        mode: "insensitive",
                    },
                },
            });
            return {
                hotels,
                currentPage: page,
                totalPages: Math.ceil(totalHotels / limit),
                totalItems: totalHotels,
            };
        }
        catch (error) {
            console.error("Error fetching hotels:", error);
            throw new Error("Failed to fetch hotels");
        }
    }
    async getHotelById(id) {
        try {
            return await prisma.hotels.findUnique({
                where: { id },
                include: {
                    rooms: true, // Include associated rooms
                    price: true,
                    supplier: true,
                },
            });
        }
        catch (error) {
            console.error(`Error fetching hotel with ID ${id}:`, error);
            throw new Error("Failed to fetch hotel");
        }
    }
    async updateHotel(id, data) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        try {
            return await prisma.hotels.update({
                where: { id },
                data: {
                    name: (_a = data.name) !== null && _a !== void 0 ? _a : "",
                    location: (_b = data.location) !== null && _b !== void 0 ? _b : "",
                    starRating: (_c = data.starRating) !== null && _c !== void 0 ? _c : "",
                    tourismTax: (_d = data.tourismTax) !== null && _d !== void 0 ? _d : "",
                    stateTax: (_e = data.stateTax) !== null && _e !== void 0 ? _e : "",
                    address: (_f = data.address) !== null && _f !== void 0 ? _f : "",
                    phone: (_g = data.phone) !== null && _g !== void 0 ? _g : "",
                    description: (_h = data.description) !== null && _h !== void 0 ? _h : "",
                    termsAndConditions: (_j = data.termsAndConditions) !== null && _j !== void 0 ? _j : "",
                    supplierId: (_k = data.supplierId) !== null && _k !== void 0 ? _k : "",
                    // Correctly updating JSON fields
                    surcharge: data.surcharge ? data.surcharge : undefined,
                    closedDates: data.closedDates ? data.closedDates : undefined,
                    // Handling `price` updates
                    price: {
                        upsert: data.price
                            ? data.price.map((p) => {
                                var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                                return ({
                                    where: { id: (_a = p.id) !== null && _a !== void 0 ? _a : "" },
                                    update: {
                                        category: (_b = p.category) !== null && _b !== void 0 ? _b : "",
                                        pax: (_c = p.pax) !== null && _c !== void 0 ? _c : "",
                                        costPrice: (_d = p.costPrice) !== null && _d !== void 0 ? _d : 0,
                                        optionalPrice: (_e = p.optionalPrice) !== null && _e !== void 0 ? _e : 0,
                                    },
                                    create: {
                                        category: (_f = p.category) !== null && _f !== void 0 ? _f : "",
                                        pax: (_g = p.pax) !== null && _g !== void 0 ? _g : "",
                                        costPrice: (_h = p.costPrice) !== null && _h !== void 0 ? _h : 0,
                                        optionalPrice: (_j = p.optionalPrice) !== null && _j !== void 0 ? _j : 0,
                                    },
                                });
                            })
                            : undefined,
                    },
                },
            });
        }
        catch (error) {
            console.error(`Error updating hotel with ID ${id}:`, error);
            throw new Error("Failed to update hotel: " + error.message);
        }
    }
    async deleteHotel(id) {
        try {
            return await prisma.hotels.delete({ where: { id } });
        }
        catch (error) {
            console.error(`Error deleting hotel with ID ${id}:`, error);
            throw new Error("Failed to delete hotel");
        }
    }
    async createRoom(data) {
        const room = await prisma.room.create({
            data: {
                name: data.name,
                hotelId: data.hotelId,
                maxPax: data.maxPax,
                extraBeds: data.extraBeds,
                prices: {
                    create: data.prices.map((price) => ({
                        category: price.category,
                        pax: price.pax,
                        costPrice: price.costPrice,
                        optionalPrice: price.optionalPrice,
                    })),
                },
            },
            include: {
                prices: true,
                hotel: true,
            },
        });
        console.log(room);
        return room;
    }
    async updateRoom(id, data) {
        return await prisma.room.update({
            where: { id },
            data: {
                name: data.name,
                maxPax: data.maxPax,
                extraBeds: data.extraBeds,
                prices: {
                    create: data.prices.filter((price) => !price.id).map((price) => ({
                        category: price.category,
                        pax: price.pax,
                        costPrice: price.costPrice,
                        optionalPrice: price.optionalPrice,
                    })),
                    update: data.prices.filter((price) => price.id).map((price) => ({
                        where: { id: price.id },
                        data: {
                            category: price.category,
                            pax: price.pax,
                            costPrice: price.costPrice,
                            optionalPrice: price.optionalPrice,
                        },
                    })),
                },
            },
            include: {
                prices: true,
                hotel: true,
            },
        });
    }
    async deleteRoom(id) {
        return await prisma.room.delete({ where: { id } });
    }
    async getRoomById(id) {
        return await prisma.room.findUnique({
            where: { id },
            include: {
                prices: true,
                hotel: true,
            },
        });
    }
    async getRoomsByHotelId(hotelId) {
        return await prisma.room.findMany({
            where: { hotelId },
            include: {
                prices: true,
                hotel: true,
            },
        });
    }
}
exports.HotelRepository = HotelRepository;
//# sourceMappingURL=hotelRepository.js.map