"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BookingRepository = void 0;
// backend/src/repositories/admin/bookingRepository.ts
const client_1 = require("@prisma/client");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const date_fns_1 = require("date-fns");
const convertToMYR = (price, rate) => {
    // If the price is null, undefined, or not a number, return null to avoid DB errors.
    if (price === null || price === undefined || isNaN(Number(price))) {
        return null;
    }
    // The rate is how many MYR one unit of the foreign currency is worth.
    // So, we multiply the foreign currency price by the rate to get the MYR value.
    return Number(price) * rate;
};
class BookingRepository {
    async fetchQuotationWithDetails(quotationId) {
        try {
            return await prismaClient_1.default.quotation.findUnique({
                where: { id: quotationId },
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                    Agent: true, // Include agent details if needed
                },
            });
        }
        catch (error) {
            console.error("Error fetching quotation details:", error);
            throw new Error("Failed to fetch quotation details");
        }
    }
    async handleCreateBooking(quotation) {
        var _a, _b, _c, _d, _e, _f;
        try {
            console.log("Quotation received for booking creation:", quotation);
            if (!quotation)
                throw new Error("Quotation data is missing.");
            // Determine the conversion rate. If currency is MYR, rate is 1.
            // Otherwise, use the provided rate.
            const conversionRate = (quotation.currency !== 'MYR' && quotation.currencyRate)
                ? Number(quotation.currencyRate)
                : 1;
            // Add a safeguard for an invalid rate
            if (isNaN(conversionRate) || conversionRate <= 0) {
                throw new Error(`Invalid currency rate for ${quotation.currency}: ${quotation.currencyRate}`);
            }
            const booking = await prismaClient_1.default.booking.create({
                data: {
                    // Booking-level data
                    reference: quotation.refNo.replace(/^QT-/, 'BK-'),
                    leadNo: quotation.leadRefNo,
                    quotationId: quotation.id,
                    travelDates: quotation.travelDates,
                    agent: (_a = quotation.leadAgent) !== null && _a !== void 0 ? _a : null,
                    guestName: (_b = quotation.guestName) !== null && _b !== void 0 ? _b : null,
                    // ** IMPORTANT: Set new booking currency to MYR after conversion **
                    currency: 'MYR',
                    currencyRate: 1,
                    // Convert top-level prices to MYR
                    quotePrice: (_c = convertToMYR(quotation.totalAfterMarkup, conversionRate)) !== null && _c !== void 0 ? _c : 0,
                    actualCost: (_d = convertToMYR(quotation.totalBeforeMarkup, conversionRate)) !== null && _d !== void 0 ? _d : 0,
                    actualPrice: 0, // Stays 0 initially
                    profit: 0, // Stays 0 initially
                    // Nested accommodations with converted prices
                    accommodations: {
                        create: ((_e = quotation.accommodations) !== null && _e !== void 0 ? _e : []).map((acc) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
                            return ({
                                // Non-price fields remain the same
                                hotel: (_a = acc.hotel) !== null && _a !== void 0 ? _a : null,
                                hotelAddress: (_b = acc.hotelAddress) !== null && _b !== void 0 ? _b : null,
                                supplierId: (_c = acc.supplierId) !== null && _c !== void 0 ? _c : null,
                                supplierName: (_d = acc.supplierName) !== null && _d !== void 0 ? _d : null,
                                supplierContact: (_e = acc.supplierContact) !== null && _e !== void 0 ? _e : null,
                                details: (_f = acc.details) !== null && _f !== void 0 ? _f : null,
                                confirmationNo: (_g = acc.confirmationNo) !== null && _g !== void 0 ? _g : null,
                                room: (_h = acc.room) !== null && _h !== void 0 ? _h : null,
                                travelDate: acc.travelDate ? new Date(acc.travelDate) : null,
                                endDate: acc.endDate ? new Date(acc.endDate) : null,
                                nights: (_j = acc.nights) !== null && _j !== void 0 ? _j : null,
                                surchargeNights: (_k = acc.surchargeNights) !== null && _k !== void 0 ? _k : null,
                                quantity: (_l = acc.quantity) !== null && _l !== void 0 ? _l : null,
                                quotationId: quotation.id,
                                // Convert accommodation-level price to MYR
                                quotePrice: convertToMYR(acc.totalPriceAfterMarkup, conversionRate),
                                priceDetailsAccommodation: {
                                    create: ((_m = acc.priceDetailsAccommodation) !== null && _m !== void 0 ? _m : []).map((pd) => {
                                        var _a, _b, _c, _d;
                                        return ({
                                            roomId: (_a = pd.roomId) !== null && _a !== void 0 ? _a : null,
                                            category: (_b = pd.category) !== null && _b !== void 0 ? _b : null,
                                            pax: (_c = pd.pax) !== null && _c !== void 0 ? _c : null,
                                            quantity: (_d = pd.quantity) !== null && _d !== void 0 ? _d : null,
                                            // Convert all nested price details to MYR
                                            price: convertToMYR(pd.price, conversionRate),
                                            quotePrice: convertToMYR(pd.priceAfterMarkup, conversionRate),
                                            costPrice: convertToMYR(pd.costPrice, conversionRate),
                                            optionalPrice: convertToMYR(pd.optionalPrice, conversionRate),
                                            totalPrice: convertToMYR(pd.totalPrice, conversionRate),
                                            totalPriceAfterMarkup: convertToMYR(pd.totalPriceAfterMarkup, conversionRate),
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                    // Nested itineraries with converted prices
                    itineraries: {
                        create: ((_f = quotation.itineraries) !== null && _f !== void 0 ? _f : []).map((itin) => {
                            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                            return ({
                                // Non-price fields remain the same
                                quotationId: quotation.id,
                                supplierId: (_a = itin.supplierId) !== null && _a !== void 0 ? _a : null,
                                supplierName: (_b = itin.supplierName) !== null && _b !== void 0 ? _b : null,
                                supplierContact: (_c = itin.supplierContact) !== null && _c !== void 0 ? _c : null,
                                details: (_d = itin.details) !== null && _d !== void 0 ? _d : null,
                                confirmationNo: (_e = itin.confirmationNo) !== null && _e !== void 0 ? _e : null,
                                date: itin.date ? new Date(itin.date) : null,
                                label: (_f = itin.label) !== null && _f !== void 0 ? _f : null,
                                category: (_g = itin.category) !== null && _g !== void 0 ? _g : null,
                                guideId: (_h = itin.guideId) !== null && _h !== void 0 ? _h : null,
                                priceDetails: (_j = itin.priceDetails) !== null && _j !== void 0 ? _j : client_1.Prisma.JsonNull,
                                // Convert itinerary-level price to MYR
                                quotePrice: convertToMYR(itin.totalPriceAfterMarkup, conversionRate),
                                priceDetailsList: {
                                    create: ((_k = itin.priceDetailsList) !== null && _k !== void 0 ? _k : []).map((pd) => {
                                        var _a, _b;
                                        return ({
                                            category: (_a = pd.category) !== null && _a !== void 0 ? _a : null,
                                            quantity: (_b = pd.quantity) !== null && _b !== void 0 ? _b : null,
                                            // Convert all nested price details to MYR
                                            price: convertToMYR(pd.price, conversionRate),
                                            quotePrice: convertToMYR(pd.totalPriceAfterMarkup, conversionRate), // As per original structure
                                            totalPrice: convertToMYR(pd.totalPrice, conversionRate),
                                            totalPriceAfterMarkup: convertToMYR(pd.totalPriceAfterMarkup, conversionRate),
                                        });
                                    }),
                                },
                            });
                        }),
                    },
                },
                // Include relations in the returned object
                include: {
                    lead: true,
                    accommodations: { include: { priceDetailsAccommodation: true } },
                    itineraries: { include: { priceDetailsList: true } },
                },
            });
            console.log("Booking created successfully in MYR:", booking);
            return booking;
        }
        catch (error) {
            console.error("Error creating booking with currency conversion:", error);
            // Be specific about the error source if possible
            if (error instanceof Error) {
                throw new Error(`Failed to create booking from quotation: ${error.message}`);
            }
            else {
                throw new Error("Failed to create booking from quotation: Unknown error");
            }
        }
    }
    async getAllBookings() {
        try {
            return await prismaClient_1.default.booking.findMany({
                include: {
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: true,
                },
                orderBy: {
                    updatedAt: 'desc',
                },
            });
        }
        catch (error) {
            console.error("Error fetching all bookings:", error);
            throw new Error("Failed to retrieve quotations");
        }
    }
    async getBookingById(id) {
        try {
            const booking = await prismaClient_1.default.booking.findUnique({
                where: { id },
                include: {
                    hotelVouchers: true,
                    accommodations: {
                        include: { priceDetailsAccommodation: true },
                    },
                    itineraries: {
                        include: { priceDetailsList: true },
                    },
                    lead: {
                        include: {
                            Agent: true,
                        },
                    },
                },
            });
            if (!booking) {
                return null;
            }
            return booking;
        }
        catch (error) {
            console.error("Error fetching Booking:", error);
            throw error;
        }
    }
    async findBookingWithAccommodations(bookingId) {
        return prismaClient_1.default.booking.findUnique({
            where: { id: bookingId },
            include: { accommodations: true }, // This will include BookingAccommodation records
        });
    }
    // This is the full updateAccommodation method from your first snippet
    // Assuming necessary imports like `Prisma`, `addDays`, `format`, `HotelVoucher` are already in place.
    // ... inside your repository class ...
    async updateAccommodation(accommodationId, updateData // Using the defined type
    ) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        try {
            const updatedAccommodation = await prismaClient_1.default.bookingAccommodation.update({
                where: { id: accommodationId },
                data: {
                    hotel: updateData.hotel,
                    room: updateData.room,
                    supplierName: updateData.supplierName,
                    supplierContact: updateData.supplierContact,
                    travelDate: updateData.travelDate ? new Date(updateData.travelDate) : undefined,
                    nights: updateData.nights,
                    quotePrice: updateData.quotePrice,
                    actualPrice: updateData.actualPrice,
                    confirmationNo: updateData.confirmationNo,
                    details: updateData.details,
                    quantity: updateData.quantity,
                },
            });
            if (!updatedAccommodation) {
                throw new Error("Accommodation not found or failed to update.");
            }
            // This part remains the same: update price details if they are provided
            if (updateData.priceDetailsAccommodation && updateData.priceDetailsAccommodation.length > 0) {
                await prismaClient_1.default.bookingPriceDetailsAccommodation.deleteMany({
                    where: { accommodationId: accommodationId }
                });
                await prismaClient_1.default.bookingPriceDetailsAccommodation.createMany({
                    data: updateData.priceDetailsAccommodation.map(detail => ({
                        ...detail,
                        accommodationId: accommodationId,
                    })),
                });
            }
            // --- START: MODIFIED LOGIC FOR VOUCHER DATA ---
            // Re-fetch the accommodation WITH its relations AFTER price details have been updated.
            // This is crucial to get the latest data for voucher creation.
            const accommodationWithRelations = await prismaClient_1.default.bookingAccommodation.findUnique({
                where: { id: accommodationId },
                include: {
                    booking: {
                        select: {
                            id: true,
                            reference: true,
                            leadNo: true,
                            lead: {
                                select: {
                                    customerName: true,
                                    totalPax: true,
                                    adults: true,
                                    kids: true,
                                }
                            }
                        },
                    },
                    hotelVoucher: true,
                    priceDetailsAccommodation: true, // IMPORTANT: Include the price details
                }
            });
            if (!accommodationWithRelations) {
                throw new Error("Failed to re-fetch accommodation after update.");
            }
            // --- NEW VOUCHER PARSING LOGIC ---
            let mealPlanItems = new Set();
            let specialInclusionItems = [];
            let roomDetailsDescription = `${((_a = accommodationWithRelations.priceDetailsAccommodation[0]) === null || _a === void 0 ? void 0 : _a.quantity) || 1} X ${accommodationWithRelations.room || 'Standard Room'}`;
            // Iterate through the structured price details
            if (accommodationWithRelations.priceDetailsAccommodation) {
                accommodationWithRelations.priceDetailsAccommodation.forEach(detail => {
                    if (detail.category) {
                        const categoryLower = detail.category.toLowerCase();
                        // Check for meal plan keywords
                        if (categoryLower.includes('breakfast')) {
                            mealPlanItems.add('Breakfast');
                        }
                        if (categoryLower.includes('child breakfast')) {
                            mealPlanItems.add('Child Breakfast');
                        }
                        // Add more meal plan checks here if needed (e.g., HB, FB)
                        // Check for special inclusion keywords
                        if (categoryLower.includes('extra bed')) {
                            // You can format this string however you like
                            specialInclusionItems.push(`${detail.quantity || 1} x Extra Bed (${detail.pax || ''} )`);
                        }
                        // Add more special inclusion checks here if needed (e.g., honeymoon setup)
                    }
                });
            }
            // Also check the general `details` field as a fallback
            if (accommodationWithRelations.details) {
                const detailsLower = accommodationWithRelations.details.toLowerCase();
                if (detailsLower.includes("bb") || detailsLower.includes("breakfast"))
                    mealPlanItems.add("Breakfast");
                else if (detailsLower.includes("hb") || detailsLower.includes("child breakfast"))
                    mealPlanItems.add("Child Breakfast");
                else if (detailsLower.includes("fb") || detailsLower.includes("full board"))
                    mealPlanItems.add("FB");
                else if (detailsLower.includes("ai") || detailsLower.includes("all inclusive"))
                    mealPlanItems.add("AI");
            }
            // Finalize the strings
            const finalMealPlan = mealPlanItems.size > 0 ? Array.from(mealPlanItems).join(' & ') : "N/A";
            const finalSpecialInclusions = specialInclusionItems.length > 0 ? specialInclusionItems.join(', ').toUpperCase() : "N/A";
            // Build room description
            if (finalMealPlan !== "N/A") {
                roomDetailsDescription += ` WITH ${finalMealPlan}`;
            }
            roomDetailsDescription += ` X ${accommodationWithRelations.nights || 0} Nights`;
            // --- END: NEW VOUCHER PARSING LOGIC ---
            const checkInDate = accommodationWithRelations.travelDate ? new Date(accommodationWithRelations.travelDate) : new Date();
            const nights = accommodationWithRelations.nights || 0;
            const checkOutDateObj = (0, date_fns_1.addDays)(checkInDate, nights);
            const hotelVoucherDataForDb = {
                customerName: ((_c = (_b = accommodationWithRelations.booking) === null || _b === void 0 ? void 0 : _b.lead) === null || _c === void 0 ? void 0 : _c.customerName) || 'N/A',
                totalAdults: ((_e = (_d = accommodationWithRelations.booking) === null || _d === void 0 ? void 0 : _d.lead) === null || _e === void 0 ? void 0 : _e.adults) || 0,
                totalKids: ((_g = (_f = accommodationWithRelations.booking) === null || _f === void 0 ? void 0 : _f.lead) === null || _g === void 0 ? void 0 : _g.kids) || 0,
                hotelName: accommodationWithRelations.hotel || 'N/A Hotel',
                hotelAddress: accommodationWithRelations.hotelAddress || 'Hotel Address Not Provided',
                hotelPhone: accommodationWithRelations.supplierContact || 'Hotel Phone Not Provided',
                checkInDate: checkInDate,
                checkOutDate: checkOutDateObj,
                roomDetails: roomDetailsDescription,
                mealPlan: finalMealPlan, // Use the newly parsed meal plan
                confirmationNumber: accommodationWithRelations.confirmationNo || 'N/A',
                specialInclusions: finalSpecialInclusions, // Use the newly parsed inclusions
                termsAndConditions: "Standard hotel terms apply. Please refer to hotel policy.",
            };
            // --- The rest of your voucher creation/update logic remains the same ---
            console.log("Hotel Voucher data prepared for DB:", hotelVoucherDataForDb);
            let persistedHotelVoucher;
            const uniqueVoucherNo = `HV-${((_j = (_h = accommodationWithRelations.booking) === null || _h === void 0 ? void 0 : _h.reference) === null || _j === void 0 ? void 0 : _j.replace('BK-', '')) || 'DEFREF'}-${accommodationWithRelations.id.slice(-4).toUpperCase()}`;
            const bookingIdForVoucher = accommodationWithRelations.bookingId;
            if (accommodationWithRelations.hotelVoucher) {
                persistedHotelVoucher = await prismaClient_1.default.hotelVoucher.update({
                    where: { id: accommodationWithRelations.hotelVoucher.id },
                    data: {
                        ...hotelVoucherDataForDb,
                        ...(bookingIdForVoucher && { booking: { connect: { id: bookingIdForVoucher } } }),
                    },
                });
                console.log("HotelVoucher updated:", persistedHotelVoucher.id);
            }
            else {
                persistedHotelVoucher = await prismaClient_1.default.hotelVoucher.create({
                    data: {
                        ...hotelVoucherDataForDb,
                        voucherNo: uniqueVoucherNo,
                        accommodation: { connect: { id: accommodationId } },
                        ...(bookingIdForVoucher && { booking: { connect: { id: bookingIdForVoucher } } }),
                    },
                });
                console.log("HotelVoucher created:", persistedHotelVoucher.id);
            }
            // ... your frontendVoucherObject creation and final return ...
            const frontendVoucherObject = {
                id: persistedHotelVoucher.id,
                voucherNumber: persistedHotelVoucher.voucherNo,
                type: 'Hotel',
                serviceName: persistedHotelVoucher.hotelName,
                customerName: persistedHotelVoucher.customerName || 'N/A',
                totalAdults: persistedHotelVoucher.totalAdults || 0,
                totalKids: persistedHotelVoucher.totalKids || 0,
                hotelAddress: persistedHotelVoucher.hotelAddress || 'N/A',
                hotelPhone: persistedHotelVoucher.hotelPhone || 'N/A',
                confirmationNumber: persistedHotelVoucher.confirmationNumber || 'N/A',
                checkInDate: (0, date_fns_1.format)(new Date(persistedHotelVoucher.checkInDate), 'dd.MM.yyyy'),
                checkOutDate: (0, date_fns_1.format)(new Date(persistedHotelVoucher.checkOutDate), 'dd.MM.yyyy'),
                nights: nights,
                roomDetailsDescription: persistedHotelVoucher.roomDetails,
                mealPlan: persistedHotelVoucher.mealPlan || 'N/A',
                specialInclusions: persistedHotelVoucher.specialInclusions || 'N/A',
                status: 'Confirmed',
            };
            return {
                updatedAccommodation: accommodationWithRelations, // Return the object with all relations
                voucher: frontendVoucherObject,
            };
        }
        catch (error) {
            console.error("Error in updateAccommodationAndManageHotelVoucher:", error);
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError && error.code === 'P2002') {
                throw new Error("A voucher with this number already exists.");
            }
            throw error;
        }
    }
    async updateItinerary(itineraryId, updateData) {
        try {
            const existing = await prismaClient_1.default.bookingItinerary.findUnique({
                where: { id: itineraryId },
                include: { priceDetailsList: true }
            });
            if (!existing) {
                throw new Error("Itinerary not found");
            }
            await prismaClient_1.default.bookingItinerary.update({
                where: { id: itineraryId },
                data: {
                    date: updateData.date ? new Date(updateData.date) : undefined,
                    label: updateData.label,
                    category: updateData.category,
                    supplierId: updateData.supplierId,
                    supplierName: updateData.supplierName,
                    supplierContact: updateData.supplierContact,
                    quotePrice: updateData.quotePrice, // Decimal
                    actualPrice: updateData.actualPrice, // Decimal
                    confirmationNo: updateData.confirmationNo, // Schema uses confirmationNo
                    details: updateData.details,
                },
            });
            if (updateData.priceDetailsList) {
                await prismaClient_1.default.bookingPriceDetailItinerary.deleteMany({
                    where: { itineraryId: itineraryId }
                });
                await prismaClient_1.default.bookingPriceDetailItinerary.createMany({
                    data: updateData.priceDetailsList.map(detail => ({
                        ...detail,
                        itineraryId: itineraryId,
                    }))
                });
            }
            return await prismaClient_1.default.bookingItinerary.findUnique({
                where: { id: itineraryId },
                include: { priceDetailsList: true }
            });
        }
        catch (error) {
            console.error("Error updating itinerary:", error);
            throw error;
        }
    }
    async getFinalItineraryItemsByBookingId(bookingId) {
        return prismaClient_1.default.finalItineraryItem.findMany({
            where: { bookingId },
            orderBy: [
                { day: 'asc' },
                { time: 'asc' },
            ],
        });
    }
    async createFinalItineraryItem(bookingId, data) {
        return prismaClient_1.default.finalItineraryItem.create({
            data: {
                ...data,
                bookingId,
            },
        });
    }
    async updateFinalItineraryItem(itemId, data) {
        return prismaClient_1.default.finalItineraryItem.update({
            where: { id: itemId },
            data,
        });
    }
    async deleteFinalItineraryItem(itemId) {
        return prismaClient_1.default.finalItineraryItem.delete({
            where: { id: itemId },
        });
    }
    async generateInitialFinalItineraryForBooking(bookingId) {
        const booking = await prismaClient_1.default.booking.findUnique({
            where: { id: bookingId },
            include: {
                accommodations: true, // This includes BookingAccommodation model which has hotelAddress
                itineraries: true,
            },
        });
        if (!booking) {
            throw new Error('Booking not found');
        }
        const itemsToCreate = [];
        const getDayNumber = (itemDate) => {
            if (!itemDate || !booking.travelDates)
                return 1;
            const tripStartDateStr = booking.travelDates.split(" to ")[0];
            const tripStart = new Date(tripStartDateStr);
            const currentItemDate = new Date(itemDate);
            if (isNaN(tripStart.getTime()) || isNaN(currentItemDate.getTime()))
                return 1;
            const diffTime = Math.abs(currentItemDate.setHours(0, 0, 0, 0) - tripStart.setHours(0, 0, 0, 0));
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            return diffDays + 1;
        };
        // booking.accommodations is an array of BookingAccommodation
        booking.accommodations.forEach(acc => {
            if (acc.hotel && acc.travelDate) {
                itemsToCreate.push({
                    day: getDayNumber(acc.travelDate),
                    activity: `Check-in: ${acc.hotel}`,
                    location: acc.hotelAddress || acc.hotel || null, // acc.hotelAddress should exist here
                    time: '14:00',
                    sourceType: 'ACCOMMODATION',
                    sourceId: acc.id,
                    notes: acc.details || null
                });
            }
            if (acc.hotel && acc.endDate) {
                itemsToCreate.push({
                    day: getDayNumber(acc.endDate),
                    activity: `Check-out: ${acc.hotel}`,
                    location: acc.hotelAddress || acc.hotel || null, // acc.hotelAddress should exist here
                    time: '12:00',
                    sourceType: 'ACCOMMODATION',
                    sourceId: acc.id,
                    notes: null
                });
            }
        });
        booking.itineraries.forEach(itin => {
            if (itin.label && itin.date) {
                itemsToCreate.push({
                    day: getDayNumber(itin.date),
                    activity: itin.label,
                    location: itin.category || null,
                    time: '10:00',
                    sourceType: 'BOOKING_ITINERARY',
                    sourceId: itin.id,
                    notes: itin.details || null
                });
            }
        });
        if (itemsToCreate.length > 0) {
            await prismaClient_1.default.finalItineraryItem.createMany({
                data: itemsToCreate.map(item => ({ ...item, bookingId })),
                skipDuplicates: true,
            });
        }
        return this.getFinalItineraryItemsByBookingId(bookingId);
    }
}
exports.BookingRepository = BookingRepository;
//# sourceMappingURL=bookingRepository.js.map