"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AttractionRepository = void 0;
const nanoid_1 = require("nanoid");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class AttractionRepository {
    async createAttraction(data) {
        try {
            const { prices, surchargeDates, closedDates, supplierId, ...attractionData } = data;
            const createdAttraction = await prismaClient_1.default.attraction.create({
                data: {
                    id: (0, nanoid_1.nanoid)(24),
                    ...attractionData,
                    surchargeDates: surchargeDates.map(date => new Date(date)),
                    closedDates: closedDates.map(date => new Date(date)),
                    supplier: {
                        connect: {
                            id: supplierId,
                        },
                    },
                    prices: {
                        create: prices.map(price => ({
                            category: price.category,
                            paxType: price.paxType,
                            costPrice: price.costPrice,
                            optionalPrice: price.optionalPrice,
                        })),
                    },
                },
                include: {
                    prices: true,
                },
            });
            return createdAttraction;
        }
        catch (error) {
            console.error("Error creating attraction in repository:", error);
            throw new Error(error.message || "Failed to create attraction");
        }
    }
    async getAttractions() {
        try {
            return await prismaClient_1.default.attraction.findMany({
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
        }
        catch (error) {
            console.error("Error fetching attractions:", error);
            throw new Error(error.message || "Failed to fetch attractions");
        }
    }
    async getAttractionById(id) {
        try {
            return await prismaClient_1.default.attraction.findUnique({
                where: { id },
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
        }
        catch (error) {
            console.error(`Error fetching attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to fetch attraction");
        }
    }
    async updateAttraction(id, data) {
        try {
            const { prices, surchargeDates, closedDates, supplierId, ...attractionData } = data;
            // Delete existing prices
            await prismaClient_1.default.attractionPrice.deleteMany({
                where: {
                    attractionId: id,
                },
            });
            const updatedAttraction = await prismaClient_1.default.attraction.update({
                where: { id },
                data: {
                    ...attractionData,
                    surchargeDates: surchargeDates.map(date => new Date(date)),
                    closedDates: closedDates.map(date => new Date(date)),
                    supplier: {
                        connect: {
                            id: supplierId,
                        },
                    },
                    prices: {
                        create: prices.map(price => ({
                            category: price.category,
                            paxType: price.paxType,
                            costPrice: price.costPrice,
                            optionalPrice: price.optionalPrice,
                        })),
                    },
                },
                include: {
                    prices: true,
                    supplier: true, // Include supplier information
                },
            });
            return updatedAttraction;
        }
        catch (error) {
            console.error(`Error updating attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to update attraction");
        }
    }
    async deleteAttraction(id) {
        try {
            await prismaClient_1.default.attraction.delete({
                where: { id },
            });
        }
        catch (error) {
            console.error(`Error deleting attraction with ID ${id}:`, error);
            throw new Error(error.message || "Failed to delete attraction");
        }
    }
}
exports.AttractionRepository = AttractionRepository;
//# sourceMappingURL=attractionRepository.js.map