"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.authenticateUser = void 0;
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const generateJwt_1 = require("../utils/generateJwt");
require("../types/express"); // Ensure Express types are extended
const dotenv_1 = __importDefault(require("dotenv"));
dotenv_1.default.config();
const SECRET_KEY = process.env.JWT_SECRET;
const REFRESH_SECRET = process.env.JWT_REFRESH_SECRET;
const authenticateUser = async (req, res, next) => {
    try {
        const authorizationHeader = req.headers.authorization;
        let refreshTokenHeader = req.headers["x-refresh-token"];
        if (!authorizationHeader) {
            res.status(403).json({ success: false, message: "Access token is missing" });
            return;
        }
        const token = authorizationHeader.split(" ")[1];
        try {
            const decoded = jsonwebtoken_1.default.verify(token, SECRET_KEY);
            if (decoded && decoded.userId && decoded.role) {
                req.currentUser = decoded.userId;
                req.currentUserRole = decoded.role;
                req.decoded = decoded; // Store the decoded token
                return next();
            }
            res.status(403).json({ success: false, message: "Invalid token" });
            return;
        }
        catch (error) {
            if (error.name === "TokenExpiredError" && refreshTokenHeader) {
                // ✅ Ensure refreshTokenHeader is a string
                if (Array.isArray(refreshTokenHeader)) {
                    refreshTokenHeader = refreshTokenHeader[0]; // Take the first value
                }
                if (typeof refreshTokenHeader !== "string") {
                    res.status(403).json({ success: false, message: "Invalid refresh token format" });
                    return;
                }
                try {
                    const refreshDecoded = jsonwebtoken_1.default.verify(refreshTokenHeader, REFRESH_SECRET);
                    if (refreshDecoded && refreshDecoded.userId && refreshDecoded.role) {
                        req.currentUser = refreshDecoded.userId;
                        req.currentUserRole = refreshDecoded.role;
                        req.decoded = refreshDecoded;
                        const newAccessToken = (0, generateJwt_1.generateToken)(refreshDecoded.userId, refreshDecoded.role, SECRET_KEY, "1h");
                        res.setHeader("x-access-token", newAccessToken);
                        return next();
                    }
                    res.status(403).json({ success: false, message: "Invalid refresh token" });
                    return;
                }
                catch (refreshError) {
                    res.status(403).json({ success: false, message: "Invalid or expired refresh token" });
                    return;
                }
            }
            else {
                res.status(403).json({ success: false, message: "Invalid access token" });
                return;
            }
        }
    }
    catch (error) {
        console.error("Authentication Error:", error);
        res.status(403).json({ success: false, message: "Unauthorized access" });
        return;
    }
};
exports.authenticateUser = authenticateUser;
//# sourceMappingURL=authMiddleware1.js.map