"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuotationController = void 0;
const bookingService_1 = require("../../services/admin/bookingService");
class QuotationController {
    constructor(quotationService, bookingService) {
        this.quotationService = quotationService;
        this.bookingService = bookingService || new bookingService_1.BookingService();
    }
    async createQuotation(req, res) {
        try {
            const data = req.body;
            const userId = req.currentUser;
            if (!userId) {
                return res.status(401).json({ error: 'Unauthorized: User ID missing.' });
            }
            const quotation = await this.quotationService.createQuotation(data, userId);
            console.log('Quotation created successfully:', quotation);
            res.status(201).json(quotation);
        }
        catch (error) {
            console.error('Error while creating quotation:', error);
            res.status(500).json({ error: error.message || 'Internal Server Error' });
        }
    }
    async getQuotationById(req, res) {
        try {
            const { id } = req.params;
            const userId = req.currentUser;
            if (!userId) {
                return res.status(200).json([]);
            }
            if (!id) {
                return res.status(200).json([]);
            }
            const quotation = await this.quotationService.getQuotationById(id, userId);
            if (!quotation) {
                return res.status(200).json([]);
            }
            res.status(200).json(quotation);
        }
        catch (error) {
            console.error('Error fetching quotation by ID:', error);
            res.status(500).json({ error: 'Internal Server Error' });
        }
    }
    async getQuotations(req, res) {
        try {
            const userId = req.currentUser;
            console.log('Current User ID:', userId);
            if (!userId) {
                return res.status(401).json({ error: 'Unauthorized: User ID missing.' });
            }
            const quotations = await this.quotationService.getAllQuotations(userId);
            if (!quotations || quotations.length === 0) {
                return res.status(200).json([]);
            }
            res.status(200).json(quotations);
        }
        catch (error) {
            console.error('Error fetching quotations:', error);
            res.status(500).json({ error: 'Internal Server Error' });
        }
    }
    async updateQuotation(req, res) {
        try {
            const { id } = req.params;
            const data = req.body;
            const userId = req.currentUser;
            if (!userId) {
                return res.status(401).json({ error: 'Unauthorized: User ID missing.' });
            }
            if (!id) {
                return res.status(400).json({ error: 'Quotation ID is required.' });
            }
            if (!data || Object.keys(data).length === 0) {
                return res.status(400).json({ error: 'Update data is required.' });
            }
            const updatedQuotation = await this.quotationService.updateQuotation(id, data, userId);
            if (!updatedQuotation) {
                return res.status(404).json({ error: `No quotation found with ID: ${id}` });
            }
            res.status(200).json(updatedQuotation);
        }
        catch (error) {
            console.error('Error updating quotation:', error);
            res.status(500).json({ error: error.message || 'Internal Server Error' });
        }
    }
    async deleteQuotation(req, res) {
        try {
            const { id } = req.params;
            const userId = req.currentUser;
            if (!userId) {
                return res.status(200).json([]);
            }
            if (!id) {
                return res.status(200).json([]);
            }
            const deletedQuotation = await this.quotationService.deleteQuotation(id, userId);
            if (!deletedQuotation) {
                return res.status(200).json([]);
            }
            res.status(200).json({ message: 'Quotation deleted successfully.' });
        }
        catch (error) {
            console.error('Error deleting quotation:', error);
            res.status(500).json({ error: 'Internal Server Error' });
        }
    }
    async duplicateQuotation(req, res) {
        try {
            const { id } = req.params;
            const userId = req.currentUser;
            if (!userId) {
                return res.status(200).json([]);
            }
            if (!id) {
                return res.status(200).json([]);
            }
            const duplicatedQuotation = await this.quotationService.duplicateQuotation(id, userId);
            if (!duplicatedQuotation) {
                return res.status(200).json([]);
            }
            res.status(201).json(duplicatedQuotation);
        }
        catch (error) {
            console.error('Error duplicating quotation:', error);
            res.status(500).json({ error: 'Internal Server Error' });
        }
    }
    async getQuotationByLeadRefNo(req, res) {
        try {
            const { leadId } = req.params;
            if (!leadId) {
                return res.status(400).json({ error: 'Lead reference number is required.' });
            }
            const quotation = await this.quotationService.getQuotationByLeadRefNo(leadId);
            if (!quotation) {
                return res.status(200).json([]);
            }
            res.status(200).json(quotation);
        }
        catch (error) {
            console.error('Error fetching quotation by lead reference number:', error);
            res.status(500).json({ error: 'Internal Server Error' });
        }
    }
    async updateQuotationStatus(req, res) {
        try {
            const { id } = req.params;
            const { status, userName } = req.body; //userName
            const userId = req.currentUser;
            if (!userId) {
                return res.status(401).json({ error: 'Unauthorized: User ID missing.' });
            }
            if (!id || !status) {
                return res.status(400).json({ message: "Quotation ID and status are required" });
            }
            // Step 1: Update the quotation status, passing userId for authorization
            const updatedQuotation = await this.quotationService.updateQuotationStatus(id, status, userId);
            console.log("Updated quotation status in controller:", updatedQuotation);
            let booking = null;
            // Step 2: If status is Accepted, create a booking.  Consider if user should be able to do this.
            if (status === "Accepted") {
                //  booking = await this.bookingService.createBookingFromQuotation(updatedQuotation.id); // original code line
                booking = await this.bookingService.createBookingFromQuotation(updatedQuotation.id); // Pass userName for admin action logging
                console.log("Booking created from accepted quotation:", booking);
            }
            return res.status(200).json({
                message: "Quotation status updated successfully",
                quotation: updatedQuotation,
                booking: booking || null,
            });
        }
        catch (error) {
            console.error("Error in controller while updating quotation status:", error);
            return res.status(500).json({ message: error.message || "Failed to update quotation status" });
        }
    }
}
exports.QuotationController = QuotationController;
//# sourceMappingURL=userquotationController.js.map