"use strict";
// backend/src/controllers/agentController.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentController = void 0;
const notificationService_1 = require("../../services/common/notificationService");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const cloudStorageConfigFile_1 = require("../../config/Agent/cloudStorageConfigFile");
class AgentController {
    constructor(agentService) {
        this.agentService = agentService;
        this.notificationService = new notificationService_1.NotificationService(); // Initialize NotificationService
    }
    async createAgent(req, res, next) {
        try {
            // Extract agent data from request body
            const agentData = {
                agentInfo: null,
                agentCode: req.body.agentCode,
                companyName: req.body.companyName,
                contactPerson: req.body.contactPerson,
                designation: req.body.designation,
                companyAddress: req.body.companyAddress,
                state: req.body.state,
                email: req.body.email,
                phone: req.body.phone,
                markup: parseFloat(req.body.markup),
                agentPrivilege: req.body.agentPrivilege === "true", // Convert string to boolean
                // deleted: false, // Set default value for deleted
            };
            // Extract credentials data from request body
            const credentialsData = {
                username: req.body.username,
                password: req.body.password,
            };
            // Handle image upload if it exists
            if (req.file) {
                const localFilePath = req.file.path;
                const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
                const remoteFileName = `agent-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
                try {
                    await (0, cloudStorageConfigFile_1.uploadToCloud)(localFilePath, remoteFileName);
                    credentialsData.filename = cloudStorageConfigFile_1.cloudConfig.baseUrl + remoteFileName;
                    // Delete the local file after successful upload
                    fs_1.default.unlink(localFilePath, (err) => {
                        if (err) {
                            console.error('Error deleting local file:', err);
                        }
                        else {
                            console.log('Local file deleted successfully');
                        }
                    });
                }
                catch (uploadError) {
                    console.error("Cloud Upload Error:", uploadError);
                    res.status(500).json({ message: "Failed to upload image to cloud." });
                    return;
                }
            }
            // Create the agent using the agentService
            const newAgent = await this.agentService.createAgent(agentData, credentialsData);
            console.log("Controller - New Agent Created:", newAgent);
            const notification = await this.notificationService.notifyAdmin({
                type: "new_agent_created",
                title: "New Agent Registered",
                message: `Agent ${newAgent.companyName} has been added by a user.`,
                entityType: "agent",
                entityId: newAgent.id,
            });
            console.log("Notification sent:", notification);
            res.status(201).json(newAgent);
        }
        catch (error) {
            console.error("Error creating agent:", error);
            next(error);
        }
    }
    async updateAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            // Extract agent data from request body
            const agentData = {
                agentCode: req.body.agentCode,
                companyName: req.body.companyName,
                contactPerson: req.body.contactPerson,
                designation: req.body.designation,
                companyAddress: req.body.companyAddress,
                state: req.body.state,
                email: req.body.email,
                phone: req.body.phone,
                markup: parseFloat(req.body.markup),
                agentPrivilege: req.body.agentPrivilege === "true", // Convert string to boolean
                deleted: req.body.deleted === "true", // Convert string to boolean
            };
            // Extract credentials data from request body
            const credentialsData = {
                username: req.body.username,
                password: req.body.password,
            };
            // Handle image upload if it exists
            if (req.file) {
                const localFilePath = req.file.path;
                const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
                const remoteFileName = `agent-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
                try {
                    await (0, cloudStorageConfigFile_1.uploadToCloud)(localFilePath, remoteFileName);
                    credentialsData.filename = cloudStorageConfigFile_1.cloudConfig.baseUrl + remoteFileName;
                    // Delete the local file after successful upload
                    fs_1.default.unlink(localFilePath, (err) => {
                        if (err) {
                            console.error('Error deleting local file:', err);
                        }
                        else {
                            console.log('Local file deleted successfully');
                        }
                    });
                }
                catch (uploadError) {
                    console.error("Cloud Upload Error:", uploadError);
                    res.status(500).json({ message: "Failed to upload image to cloud." });
                    return;
                }
            }
            // Update the agent using the agentService
            const updatedAgent = await this.agentService.updateAgent(agentId, agentData, credentialsData);
            if (!updatedAgent) {
                res.status(404).json({ message: "Agent not found" });
                return;
            }
            res.json(updatedAgent);
        }
        catch (error) {
            console.error("Error updating agent:", error);
            next(error);
        }
    }
    async getAllAgents(req, res, next) {
        try {
            const agents = await this.agentService.getAllAgents();
            res.json(agents);
        }
        catch (error) {
            console.error("Error getting agents:", error);
            next(error);
        }
    }
    async softDeleteAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            console.log("Controller - Soft Deleting Agent:", agentId);
            // Call the service to soft delete the agent
            const deletedAgent = await this.agentService.softDeleteAgent(agentId);
            if (!deletedAgent) {
                res.status(404).json({ message: "Agent not found" });
                return;
            }
            res.json(deletedAgent);
        }
        catch (error) {
            console.error("Error deleting agent:", error);
            next(error);
        }
    }
    async restoreAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            const restoredAgent = await this.agentService.restoreAgent(agentId);
            if (!restoredAgent) {
                res.status(404).json({ message: "Agent not found for restore" });
                return;
            }
            res.json(restoredAgent);
        }
        catch (error) {
            console.error("Error restoring agent:", error);
            next(error);
        }
    }
    async deleteAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            await this.agentService.deleteAgent(agentId);
            res.status(204).send();
        }
        catch (error) {
            console.error("Error deleting agent:", error);
            next(error);
        }
    }
}
exports.AgentController = AgentController;
//# sourceMappingURL=userAgentController.js.map