"use strict";
// File: backend/src/controllers/user/documentController.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DocumentController = void 0;
const cloudStorageDocument_1 = require("../../config/cloudStorageDocument");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const nanoid_1 = require("nanoid");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
class DocumentController {
    constructor(documentService) {
        this.documentService = documentService;
    }
    async uploadDocument(req, res, next) {
        var _a, _b, _c;
        try {
            const leadId = req.params.leadId;
            if (!req.file) {
                res.status(400).json({ message: 'No file uploaded' });
                return;
            }
            // Sanitize filename
            const sanitizedFilename = path_1.default.basename(req.file.originalname);
            // Get file extension
            const fileExtension = path_1.default.extname(sanitizedFilename).toLowerCase();
            // Generate unique remote filename
            const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
            const remoteFileName = `document-${uniqueSuffix}${fileExtension}`;
            // Construct local file path
            const localFilePath = req.file.path;
            try {
                // Upload to Filezilla
                await (0, cloudStorageDocument_1.uploadToCloudDocument)(localFilePath, remoteFileName);
                const fileUrl = cloudStorageDocument_1.cloudConfigDocument.baseUrl + remoteFileName;
                // Delete the local file after successful upload
                fs_1.default.unlink(localFilePath, (err) => {
                    if (err) {
                        console.error('Error deleting local file:', err);
                    }
                    else {
                        console.log('Local file deleted successfully');
                    }
                });
                let uploadedBy = 'Unknown User';
                if (((_a = req.user) === null || _a === void 0 ? void 0 : _a.role) === 'user') {
                }
                else {
                    try {
                        // Fetch user information based on ID
                        const user = await prismaClient_1.default.user.findUnique({
                            where: { id: (_b = req.user) === null || _b === void 0 ? void 0 : _b.id }
                        });
                        if (user && user.name) {
                            uploadedBy = user.name;
                        }
                        else {
                            console.warn('User not found or missing name with id:', (_c = req.user) === null || _c === void 0 ? void 0 : _c.id);
                        }
                    }
                    catch (error) {
                        console.error('Error fetching user data:', error);
                        uploadedBy = 'User'; // Fallback in case of error
                    }
                }
                // Save document metadata to the database
                const documentData = {
                    id: (0, nanoid_1.nanoid)(),
                    name: sanitizedFilename,
                    type: req.body.type,
                    fileUrl: fileUrl,
                    fileType: fileExtension.slice(1),
                    size: (req.file.size / 1024 / 1024).toFixed(2) + ' MB',
                    uploadedOn: new Date(),
                    uploadedBy: uploadedBy,
                    leadId: leadId,
                };
                const newDocument = await this.documentService.createDocument(documentData);
                res.status(201).json(newDocument);
            }
            catch (uploadError) {
                console.error("Filezilla Upload Error:", uploadError);
                res.status(500).json({ message: "Failed to upload file to Filezilla." });
            }
        }
        catch (error) {
            console.error('Error uploading document:', error);
            next(error);
        }
    }
    async getDocumentsForLead(req, res, next) {
        try {
            const leadId = req.params.leadId;
            const documents = await this.documentService.getDocumentsForLead(leadId);
            res.json(documents);
        }
        catch (error) {
            console.error('Error getting documents for lead:', error);
            next(error);
        }
    }
    async deleteDocument(req, res, next) {
        try {
            const documentId = req.params.id;
            await this.documentService.deleteDocument(documentId);
            res.status(204).send(); // 204 No Content - success, no content to return
        }
        catch (error) {
            console.error('Error deleting document:', error);
            next(error);
        }
    }
}
exports.DocumentController = DocumentController;
//# sourceMappingURL=documentController.js.map