"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BookingController = void 0;
const UserBookingService_1 = require("../../services/user/UserBookingService");
class BookingController {
    constructor(bookingService) {
        this.bookingService = bookingService || new UserBookingService_1.BookingService();
    }
    async fetchQuotationWithDetails(req, res) {
        try {
            const { id } = req.params;
            if (!id) {
                return res.status(400).json({ error: "Quotation ID is required" });
            }
            const quotationDetails = await this.bookingService.fetchQuotationWithDetails(id);
            return res.status(200).json(quotationDetails);
        }
        catch (error) {
            console.error("Error fetching quotation details:", error);
            return res.status(500).json({ error: error.message || "Failed to fetch quotation details" });
        }
    }
    async createBooking(req, res) {
        try {
            const { quotationId } = req.body;
            if (!quotationId) {
                return res.status(400).json({ error: "quotationId is required" });
            }
            const booking = await this.bookingService.createBookingFromQuotation(quotationId);
            return res.status(201).json(booking);
        }
        catch (error) {
            console.error("Error creating booking:", error);
            return res.status(500).json({ error: error.message || "Failed to create booking" });
        }
    }
    async getBookings(req, res) {
        try {
            const userId = req.currentUser; // Get user ID from authentication middleware
            const bookings = await this.bookingService.getAllBookings(userId);
            return res.status(200).json(bookings || []);
        }
        catch (error) {
            console.error("Error fetching quotations:", error);
            res.status(500).json({ error: "Internal Server Error" });
        }
    }
    async getBookingById(req, res) {
        try {
            const { id } = req.params;
            console.log("Fetching quotation for ID:", id);
            if (!id) {
                return res.status(400).json({ error: "Booking ID is required" });
            }
            const booking = await this.bookingService.getBookingById(id);
            console.log(booking, "Booking in controller");
            if (!booking) {
                return res.status(404).json({ error: "Quotation not found" });
            }
            res.status(200).json(booking);
        }
        catch (error) {
            console.error("Controller error:", error);
            res.status(500).json({ error: "Internal Server Error" });
        }
    }
    async updateBookingStatus(req, res) {
        try {
            const { id } = req.params;
            const { status, userName } = req.body;
            if (!id) {
                return res.status(400).json({ error: "Booking ID is required" });
            }
            if (!status) {
                return res.status(400).json({ error: "New status is required" });
            }
            const updatedBooking = await this.bookingService.updateBookingStatus(id, status, userName);
            res.status(200).json(updatedBooking);
        }
        catch (error) {
            console.error("Error updating booking status:", error);
            res.status(500).json({ error: error.message || "Failed to update booking status" });
        }
    }
    async updateBookingStage(req, res) {
        try {
            const { id } = req.params;
            const { stage, notes, userName } = req.body;
            if (!id) {
                return res.status(400).json({ error: "Booking ID is required" });
            }
            if (!stage) {
                return res.status(400).json({ error: "New stage is required" });
            }
            const updatedBooking = await this.bookingService.updateBookingStage(id, stage, notes, userName);
            res.status(200).json(updatedBooking);
        }
        catch (error) {
            console.error("Error updating booking stage:", error);
            res.status(500).json({ error: error.message || "Failed to update booking stage" });
        }
    }
    async updateAccommodationHandler(req, res) {
        const { bookingId, accommodationId } = req.params;
        const { hotel, room, supplierName, supplierContact, travelDate, nights, quotePrice, actualPrice, confirmationNo, details, quantity, priceDetailsAccommodation } = req.body;
        console.log("Updating accommodation with data in controller:", req.body);
        try {
            const result = await this.bookingService.updateAccommodationInBooking(bookingId, accommodationId, {
                hotel,
                room,
                supplierName,
                supplierContact,
                travelDate,
                nights,
                quotePrice,
                actualPrice,
                confirmationNo,
                details,
                quantity,
                priceDetailsAccommodation
            });
            res.status(200).json({ message: "Accommodation updated", data: result });
        }
        catch (error) {
            res.status(500).json({ error: error.message || "Internal server error" });
        }
    }
    async updateItinerary(req, res) {
        try {
            const { bookingId, itineraryId } = req.params;
            console.log("Updating itinerary with bookingId:", bookingId, "and itineraryId:", itineraryId);
            const updateData = req.body;
            const updatedItinerary = await this.bookingService.updateItinerary(bookingId, itineraryId, updateData);
            res.json({
                success: true,
                data: updatedItinerary
            });
        }
        catch (error) {
            console.error("Error updating itinerary:", error);
        }
    }
}
exports.BookingController = BookingController;
//# sourceMappingURL=UserBookingController.js.map