"use strict";
// backend/src/controllers/agentController.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentController = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const cloudStorageConfigFile_1 = require("../../config/Agent/cloudStorageConfigFile");
const HttpError_1 = require("../../utils/HttpError");
class AgentController {
    constructor(agentService) {
        this.agentService = agentService;
    }
    async createAgent(req, res, next) {
        try {
            const agentData = {
                agentInfo: null,
                agentCode: req.body.agentCode,
                companyName: req.body.companyName,
                contactPerson: req.body.contactPerson,
                designation: req.body.designation,
                companyAddress: req.body.companyAddress,
                state: req.body.state,
                email: req.body.email,
                phone: req.body.phone,
                markup: parseFloat(req.body.markup),
                agentPrivilege: req.body.agentPrivilege === "true",
            };
            const credentialsData = {
                username: req.body.username,
                password: req.body.password,
            };
            if (req.file) {
                const localFilePath = req.file.path;
                const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
                const remoteFileName = `agent-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
                await (0, cloudStorageConfigFile_1.uploadToCloud)(localFilePath, remoteFileName);
                credentialsData.filename = cloudStorageConfigFile_1.cloudConfig.baseUrl + remoteFileName;
                fs_1.default.unlink(localFilePath, (err) => err && console.error("Error deleting local file:", err));
            }
            const newAgent = await this.agentService.createAgent(agentData, credentialsData);
            res.status(201).json(newAgent);
        }
        catch (error) {
            if (error instanceof HttpError_1.HttpError) {
                res.status(error.status).json({ message: error.message });
            }
            else {
                console.error("Unexpected error creating agent:", error);
                res.status(500).json({ message: "Internal server error" });
            }
        }
    }
    async updateAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            const agentData = {
                agentCode: req.body.agentCode,
                companyName: req.body.companyName,
                contactPerson: req.body.contactPerson,
                designation: req.body.designation,
                companyAddress: req.body.companyAddress,
                state: req.body.state,
                email: req.body.email,
                phone: req.body.phone,
                markup: parseFloat(req.body.markup),
                agentPrivilege: req.body.agentPrivilege === "true",
                deleted: req.body.deleted === "true",
            };
            const credentialsData = {
                username: req.body.username,
                password: req.body.password,
            };
            if (req.file) {
                const localFilePath = req.file.path;
                const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
                const remoteFileName = `agent-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
                await (0, cloudStorageConfigFile_1.uploadToCloud)(localFilePath, remoteFileName);
                credentialsData.filename = cloudStorageConfigFile_1.cloudConfig.baseUrl + remoteFileName;
                fs_1.default.unlink(localFilePath, (err) => err && console.error("Error deleting local file:", err));
            }
            const updatedAgent = await this.agentService.updateAgent(agentId, agentData, credentialsData);
            if (!updatedAgent) {
                res.status(404).json({ message: "Agent not found" });
                return;
            }
            res.json(updatedAgent);
        }
        catch (error) {
            if (error instanceof HttpError_1.HttpError) {
                res.status(error.status).json({ message: error.message });
            }
            else {
                console.error("Unexpected error updating agent:", error);
                res.status(500).json({ message: "Internal server error" });
            }
        }
    }
    async getAllAgents(req, res, next) {
        try {
            const agents = await this.agentService.getAllAgents();
            res.json(agents);
        }
        catch (error) {
            console.error("Error getting agents:", error);
            next(error);
        }
    }
    async softDeleteAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            console.log("Controller - Soft Deleting Agent:", agentId);
            // Call the service to soft delete the agent
            const deletedAgent = await this.agentService.softDeleteAgent(agentId);
            if (!deletedAgent) {
                res.status(404).json({ message: "Agent not found" });
                return;
            }
            res.json(deletedAgent);
        }
        catch (error) {
            console.error("Error deleting agent:", error);
            next(error);
        }
    }
    async restoreAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            const restoredAgent = await this.agentService.restoreAgent(agentId);
            if (!restoredAgent) {
                res.status(404).json({ message: "Agent not found for restore" });
                return;
            }
            res.json(restoredAgent);
        }
        catch (error) {
            console.error("Error restoring agent:", error);
            next(error);
        }
    }
    async deleteAgent(req, res, next) {
        try {
            const agentId = req.params.id;
            await this.agentService.deleteAgent(agentId);
            res.status(204).send();
        }
        catch (error) {
            console.error("Error deleting agent:", error);
            next(error);
        }
    }
}
exports.AgentController = AgentController;
//# sourceMappingURL=agentController.js.map