"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentQuotationController = void 0;
class AgentQuotationController {
    constructor(quotationService) {
        this.quotationService = quotationService;
    }
    async createQuotation(req, res) {
        try {
            const data = req.body;
            const agentId = req.currentUser;
            if (!agentId) {
                return res
                    .status(401)
                    .json({ error: "Unauthorized: Agent ID missing." });
            }
            const quotation = await this.quotationService.createQuotation(data, agentId);
            console.log("Quotation created successfully:", quotation);
            res.status(201).json(quotation);
        }
        catch (error) {
            console.error("Error while creating quotation:", error);
            res
                .status(500)
                .json({ error: error.message || "Internal Server Error" });
        }
    }
    async getQuotationById(req, res) {
        try {
            const { id } = req.params;
            if (!id) {
                return res.status(400).json({ error: "Quotation ID is required." });
            }
            const quotation = await this.quotationService.getQuotationById(id);
            if (!quotation) {
                return res.status(404).json({ error: "Quotation not found." });
            }
            res.status(200).json(quotation);
        }
        catch (error) {
            console.error("Error fetching quotation by ID:", error);
            res.status(500).json({ error: "Internal Server Error" });
        }
    }
    async getQuotations(req, res) {
        try {
            const agentId = req.currentUser;
            if (!agentId) {
                return res
                    .status(401)
                    .json({ error: "Unauthorized: Agent ID missing." });
            }
            const quotations = await this.quotationService.getAllQuotations(agentId);
            if (!quotations || quotations.length === 0) {
                return res.status(200).json([]);
            }
            res.status(200).json(quotations);
        }
        catch (error) {
            console.error("Error fetching quotations:", error);
            res.status(500).json({ error: "Internal Server Error" });
        }
    }
    async updateQuotation(req, res) {
        try {
            const { id } = req.params;
            const data = req.body;
            if (!id) {
                return res.status(400).json({ error: "Quotation ID is required." });
            }
            if (!data || Object.keys(data).length === 0) {
                return res.status(400).json({ error: "Update data is required." });
            }
            const updatedQuotation = await this.quotationService.updateQuotation(id, data);
            if (!updatedQuotation) {
                return res
                    .status(404)
                    .json({ error: `No quotation found with ID: ${id}` });
            }
            res.status(200).json(updatedQuotation);
        }
        catch (error) {
            console.error("Error updating quotation:", error);
            res
                .status(500)
                .json({ error: error.message || "Internal Server Error" });
        }
    }
    async deleteQuotation(req, res) {
        try {
            const { id } = req.params;
            if (!id) {
                return res.status(400).json({ error: "Quotation ID is required." });
            }
            const deletedQuotation = await this.quotationService.deleteQuotation(id);
            if (!deletedQuotation) {
                return res.status(404).json({ error: "Quotation not found." });
            }
            res.status(200).json({ message: "Quotation deleted successfully." });
        }
        catch (error) {
            console.error("Error deleting quotation:", error);
            res.status(500).json({ error: "Internal Server Error" });
        }
    }
}
exports.AgentQuotationController = AgentQuotationController;
//# sourceMappingURL=AgentQuotationController.js.map