"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserController = void 0;
const userService_1 = require("../../services/user/userService");
const userService = new userService_1.UserService();
class UserController {
    // In UserController
    async createUser(req, res) {
        try {
            const user = await userService.createUser(req.body);
            console.log(user, "user in controller");
            res.status(201).json(user);
        }
        catch (error) {
            if (error instanceof Error) {
                if (error.message.includes("already exists")) {
                    res.status(409).json({ message: error.message }); // 409 Conflict
                }
                else {
                    res.status(400).json({ message: error.message }); // 400 Bad Request
                }
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async updateUser(req, res) {
        try {
            const user = await userService.updateUser(req.params.id, req.body);
            res.json(user);
        }
        catch (error) {
            if (error instanceof Error) {
                if (error.message.includes("already exists")) {
                    res.status(409).json({ message: error.message }); // 409 Conflict
                }
                else {
                    res.status(400).json({ message: error.message }); // 400 Bad Request
                }
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async getUsers(req, res) {
        try {
            const users = await userService.getUsers();
            res.json(users);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async getUserById(req, res) {
        try {
            const user = await userService.getUserById(req.params.id);
            if (!user) {
                res.status(404).json({ message: "User not found" });
                return;
            }
            res.json(user);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async getUserByEmail(req, res) {
        try {
            const user = await userService.getUserByEmail(req.params.email);
            if (!user) {
                res.status(404).json({ message: "User not found" });
                return;
            }
            res.json(user);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async deleteUser(req, res) {
        try {
            await userService.deleteUser(req.params.id);
            res.json({ message: "User deleted successfully" });
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async restore(req, res) {
        try {
            const userId = req.params.id;
            const restoredUser = await userService.restore(userId);
            if (!restoredUser) {
                return res.status(404).json({ message: "User not found" });
            }
            return res.status(200).json({ message: "User restored successfully" });
        }
        catch (error) {
            if (error instanceof Error) {
                return res.status(500).json({ message: error.message });
            }
            else {
                return res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async deleteUserPermanently(req, res) {
        try {
            const userId = req.params.id;
            const deletedUser = await userService.deleteUserPermanently(userId);
            if (!deletedUser) {
                return res
                    .status(404)
                    .json({ message: "User not found or already deleted" });
            }
            return res.status(200).json({ message: "User permanently deleted" });
        }
        catch (error) {
            if (error instanceof Error) {
                return res.status(500).json({ message: error.message });
            }
            else {
                return res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async getInactiveUsers(req, res) {
        try {
            const users = await userService.getInactiveUsers();
            res.json(users);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async getUsersByTeamId(req, res) {
        try {
            const teamId = req.params.id;
            console.log(teamId);
            const users = await userService.getUsersByTeamId(teamId);
            res.json(users);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
    async searchUsers(req, res) {
        try {
            const users = await userService.searchUsers(req.query.searchTerm);
            res.json(users);
        }
        catch (error) {
            if (error instanceof Error) {
                res.status(500).json({ message: error.message });
            }
            else {
                res.status(500).json({ message: "An unknown error occurred" });
            }
        }
    }
}
exports.UserController = UserController;
//# sourceMappingURL=userController.js.map