"use strict";
// backend/src/controllers/packageController.ts
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageController = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const cloudStorageConfig_1 = require("../../config/cloudStorageConfig");
class PackageController {
    constructor(packageService) {
        this.packageService = packageService;
    }
    async createPackage(req, res, next) {
        try {
            console.log("Request Body:", req.body);
            const { packageName, tagLine, briefDescription, detailedDescription, terms, notes, remarks, starRating, price, hotel, daysNights, members, hotelId, imageUrl } = req.body;
            const packageDataWithoutId = {
                packageName: packageName,
                tagLine: tagLine,
                briefDescription: briefDescription,
                detailedDescription: detailedDescription,
                terms: terms,
                notes: notes,
                remarks: remarks,
                imageUrl: imageUrl,
                starRating: parseInt(starRating),
                price: price,
                hotel: hotel,
                daysNights: daysNights,
                members: members,
                deleted: false,
                hotelId: hotelId,
            };
            const newPackage = await this.packageService.createPackage(packageDataWithoutId);
            res.status(201).json(newPackage);
            return;
        }
        catch (error) {
            console.error("Error creating package:", error);
            next(error);
            return;
        }
    }
    async uploadImage(req, res, next) {
        try {
            if (!req.file) {
                res.status(400).json({ message: "No image file provided" });
                return;
            }
            const filename = req.file.filename;
            const localFilePath = req.file.path;
            const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
            const remoteFileName = `package-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
            try {
                await (0, cloudStorageConfig_1.uploadToCloud)(localFilePath, remoteFileName);
                const cloudImageUrl = cloudStorageConfig_1.cloudConfig.baseUrl + remoteFileName;
                fs_1.default.unlink(localFilePath, (err) => {
                    if (err) {
                        console.error('Error deleting local file:', err);
                    }
                    else {
                        console.log('Local file deleted successfully');
                    }
                });
                res.status(200).json({ imageUrl: cloudImageUrl });
            }
            catch (uploadError) {
                console.error("Cloud Upload Error:", uploadError);
                res.status(500).json({ message: "Failed to upload image to cloud." });
            }
        }
        catch (error) {
            console.error("Error uploading image:", error);
            next(error);
        }
    }
    async updatePackage(req, res, next) {
        try {
            const packageId = req.params.id;
            const filename = req.file ? req.file.filename : null;
            let cloudImageUrl;
            if (filename && req.file) {
                // Construct the local file path
                const localFilePath = req.file.path;
                const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
                const remoteFileName = `package-${uniqueSuffix}${path_1.default.extname(req.file.originalname)}`;
                // Upload to the cloud
                try {
                    await (0, cloudStorageConfig_1.uploadToCloud)(localFilePath, remoteFileName);
                    cloudImageUrl = cloudStorageConfig_1.cloudConfig.baseUrl + remoteFileName;
                    // Delete the local file after successful upload
                    fs_1.default.unlink(localFilePath, (err) => {
                        if (err) {
                            console.error('Error deleting local file:', err);
                        }
                        else {
                            console.log('Local file deleted successfully');
                        }
                    });
                }
                catch (uploadError) {
                    console.error("Cloud Upload Error:", uploadError);
                    // **IMPORTANT:** Send an error response *and* `return` to prevent further execution
                    res.status(500).json({ message: "Failed to upload image to cloud." });
                    return; // <---- Add this return
                }
            }
            const packageData = {
                packageName: req.body.packageName,
                tagLine: req.body.tagLine,
                briefDescription: req.body.briefDescription,
                detailedDescription: req.body.detailedDescription,
                terms: req.body.terms,
                notes: req.body.notes,
                remarks: req.body.remarks,
                starRating: parseInt(req.body.starRating),
                price: req.body.price,
                hotel: req.body.hotel,
                daysNights: req.body.daysNights,
                members: req.body.members,
                hotelId: req.body.hotelId,
                // Only update imageUrl if a new file was uploaded.
                imageUrl: cloudImageUrl,
            };
            const updatedPackage = await this.packageService.updatePackage(packageId, packageData);
            res.json(updatedPackage);
            return;
        }
        catch (error) {
            console.error("Error updating package:", error);
            next(error);
            return;
        }
    }
    async getPackages(req, res, next) {
        try {
            const packages = await this.packageService.getAllPackages();
            res.json(packages);
            return;
        }
        catch (error) {
            console.error("Error getting packages:", error);
            next(error);
            return;
        }
    }
    async getPackageById(req, res, next) {
        try {
            const packageId = req.params.id;
            const packageItem = await this.packageService.getPackageById(packageId);
            if (!packageItem) {
                res.status(404).json({ message: "Package not found" });
                return;
            }
            console.log(packageItem);
            res.json(packageItem);
            return;
        }
        catch (error) {
            console.error("Error getting package by ID:", error);
            next(error);
            return;
        }
    }
    async deletePackage(req, res, next) {
        try {
            const packageId = req.params.id;
            await this.packageService.deletePackage(packageId);
            res.status(204).send(); // No content
            return;
        }
        catch (error) {
            console.error("Error deleting package:", error);
            next(error);
            return;
        }
    }
}
exports.PackageController = PackageController;
//# sourceMappingURL=packageController.js.map