"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LeadController = void 0;
const nanoid_1 = require("nanoid");
class LeadController {
    constructor(leadService) {
        this.leadService = leadService;
    }
    async createLead(req, res, next) {
        try {
            console.log("Request Body in Lead Controller:", req.body);
            const travelDateFrom = req.body.travelDateFrom;
            const travelDateTo = req.body.travelDateTo;
            const userName = req.body.userName;
            const leadData = {
                id: (0, nanoid_1.nanoid)(),
                empCode: req.body.empCode || null,
                agentName: req.body.agentName || null,
                agentCode: req.body.agentCode || null,
                leadNo: req.body.leadNo,
                travelDateFrom: travelDateFrom,
                travelDateTo: travelDateTo,
                totalPax: req.body.totalPax || null,
                description: req.body.description || null,
                customerName: req.body.customerName || null,
                email: req.body.email || null,
                phone: req.body.phone || null,
                adults: req.body.adults || null,
                kids: req.body.kids || null,
                infants: req.body.infants || null,
                remarks: req.body.remarks || null,
                agentId: req.body.agentId || null, // Get Agent ID from body
                userId: req.body.userId || null, // Get User ID from body
            };
            console.log("Controller Lead Data:", leadData);
            const newLead = await this.leadService.createLead(leadData, userName);
            res.status(201).json(newLead);
        }
        catch (error) {
            console.error('Error creating lead:', error);
            next(error);
        }
    }
    async getLeads(req, res, next) {
        try {
            const leads = await this.leadService.getAllLeads();
            res.json(leads);
        }
        catch (error) {
            console.error('Error getting leads:', error);
            next(error);
        }
    }
    async getLeadById(req, res, next) {
        try {
            const leadId = req.params.id;
            const lead = await this.leadService.getLeadById(leadId);
            if (!lead) {
                res.status(404).json({ message: 'Lead not found' });
                return;
            }
            res.json(lead);
        }
        catch (error) {
            console.error('Error getting lead by ID:', error);
            next(error);
        }
    }
    async updateLead(req, res, next) {
        try {
            const leadId = req.params.id;
            const travelDateFrom = req.body.travelDateFrom ? new Date(req.body.travelDateFrom).toISOString() : undefined;
            const travelDateTo = req.body.travelDateTo ? new Date(req.body.travelDateTo).toISOString() : undefined;
            const leadData = {
                empCode: req.body.empCode || null,
                agentName: req.body.agentName,
                agentCode: req.body.agentCode,
                leadNo: req.body.leadNo,
                travelDateFrom: travelDateFrom,
                travelDateTo: travelDateTo,
                totalPax: req.body.totalPax,
                description: req.body.description,
                customerName: req.body.customerName,
                email: req.body.email,
                phone: req.body.phone,
                adults: req.body.adults,
                kids: req.body.kids,
                infants: req.body.infants,
                remarks: req.body.remarks,
                agentId: req.body.agentId,
            };
            const updatedLead = await this.leadService.updateLead(leadId, leadData);
            if (updatedLead) {
                res.json(updatedLead);
            }
            else {
                res.status(404).json({ message: 'Lead not found' });
            }
        }
        catch (error) {
            console.error('Error updating lead:', error);
            next(error);
        }
    }
    async deleteLead(req, res, next) {
        try {
            const leadId = req.params.id;
            await this.leadService.deleteLead(leadId);
            res.status(204).send();
        }
        catch (error) {
            console.error('Error deleting lead:', error);
            next(error);
        }
    }
    async getLeadsByTimeframe(req, res, next) {
        try {
            const timeframe = req.params.id;
            const leadsData = await this.leadService.fetchLeads(timeframe);
            res.json(leadsData);
        }
        catch (error) {
            console.error('Error fetching leads:', error);
            next(error);
        }
    }
    async updateStatus(req, res) {
        const { id } = req.params;
        const { status } = req.body;
        try {
            const lead = await this.leadService.changeStatus(id, status);
            if (!lead)
                return res.status(404).json({ message: "Lead not found" });
            return res.status(200).json(lead);
        }
        catch (err) {
            return res.status(500).json({ message: "Server error", error: err });
        }
    }
    async updateStage(req, res) {
        const { id } = req.params;
        console.log("REq body in controller", req.body);
        const { stage, notes, userName } = req.body;
        console.log("REq body in controller", req.body);
        try {
            const lead = await this.leadService.changeStage(id, stage, userName, notes);
            if (!lead)
                return res.status(404).json({ message: "Lead not found" });
            return res.status(200).json(lead);
        }
        catch (err) {
            return res.status(500).json({ message: "Server error", error: err });
        }
    }
    async assignLeadToOperations(req, res) {
        const { leadId, operationUserId } = req.body;
        if (!leadId || !operationUserId) {
            return res.status(400).json({ message: 'leadId and operationUserId are required' });
        }
        try {
            const updatedLead = await this.leadService.assignLeadToOperations(leadId, operationUserId);
            return res.status(200).json({ message: 'Lead assigned to operations successfully', data: updatedLead });
        }
        catch (error) {
            console.error('Controller Error:', error);
            return res.status(500).json({ message: 'Failed to assign lead to operations' });
        }
    }
    async assignLeadToAssessment(req, res) {
        const { leadId, assessmentUserId } = req.body;
        if (!leadId || !assessmentUserId) {
            return res.status(400).json({ message: 'leadId and assessmentUserId are required' });
        }
        try {
            const updatedLead = await this.leadService.assignLeadToAssessment(leadId, assessmentUserId);
            return res.status(200).json({ message: 'Lead assigned to assessment successfully', data: updatedLead });
        }
        catch (error) {
            console.error('Controller Error:', error);
            return res.status(500).json({ message: 'Failed to assign lead to assessment' });
        }
    }
    async getQuotationAcceptedLeads(req, res) {
        try {
            const leads = await this.leadService.getQuotationAcceptedLeads();
            console.log("Leads in controller", leads);
            return res.status(200).json(leads);
        }
        catch (error) {
            console.error("Error fetching leads:", error);
            return res.status(500).json({ message: "Internal server error" });
        }
    }
}
exports.LeadController = LeadController;
//# sourceMappingURL=leadController.js.map