"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AttractionController = void 0;
const date_fns_1 = require("date-fns");
const prismaClient_1 = __importDefault(require("../../config/prismaClient"));
const attractionService_1 = require("../../services/admin/attractionService");
class AttractionController {
    constructor() {
        this.createAttraction = async (req, res, next) => {
            try {
                const attraction = req.body;
                // *** DATA VALIDATION HERE!! ****  IMPORTANT!!!
                if (!attraction.name || attraction.name.length < 3) {
                    return res.status(400).json({ message: "Name must be at least 3 characters long." });
                }
                if (!attraction.supplierId) {
                    return res.status(400).json({ message: "Supplier ID is required." });
                }
                // *** VALIDATE SUPPLIER ID ***
                const supplierExists = await prismaClient_1.default.suppliers.findUnique({
                    where: {
                        id: attraction.supplierId,
                    },
                });
                if (!supplierExists) {
                    return res.status(400).json({ message: "Invalid supplier ID." });
                }
                // Date validation
                if (attraction.surchargeDates) {
                    for (const dateString of attraction.surchargeDates) {
                        if (!(0, date_fns_1.isValid)(new Date(dateString))) {
                            return res.status(400).json({ message: `Invalid surcharge date: ${dateString}` });
                        }
                    }
                }
                if (attraction.closedDates) {
                    for (const dateString of attraction.closedDates) {
                        if (!(0, date_fns_1.isValid)(new Date(dateString))) {
                            return res.status(400).json({ message: `Invalid closed date: ${dateString}` });
                        }
                    }
                }
                const createdAttraction = await this.attractionService.createAttraction(attraction);
                res.status(201).json(createdAttraction);
            }
            catch (error) {
                console.error("Error creating attraction in controller:", error);
                res.status(500).json({ message: error.message || "Internal Server Error" });
            }
        };
        this.getAttractions = async (req, res, next) => {
            try {
                const attractions = await this.attractionService.getAttractions();
                res.json(attractions);
            }
            catch (error) {
                next(error);
            }
        };
        this.getAttractionById = async (req, res, next) => {
            try {
                const attraction = await this.attractionService.getAttractionById(req.params.id);
                if (!attraction) {
                    res.status(404).json({ message: "Attraction not found" });
                    return;
                }
                res.json(attraction);
            }
            catch (error) {
                next(error);
            }
        };
        this.updateAttraction = async (req, res, next) => {
            try {
                const attraction = await this.attractionService.updateAttraction(req.params.id, req.body);
                res.json(attraction);
            }
            catch (error) {
                next(error);
            }
        };
        this.deleteAttraction = async (req, res, next) => {
            try {
                await this.attractionService.deleteAttraction(req.params.id);
                res.json({ message: "Attraction deleted successfully" });
            }
            catch (error) {
                next(error);
            }
        };
        this.attractionService = new attractionService_1.AttractionService();
    }
}
exports.AttractionController = AttractionController;
//# sourceMappingURL=attractionController.js.map